#!/usr/bin/env python
# Stricken host log gathering system
# Author: Nicholas Kurjanowicz
import os
import argparse
import logging
import time
from shutil import copy2, copytree, rmtree
from sys import exit

# instantiate logging
logging.basicConfig()
log = logging.getLogger('log-gather')

TMP_DIR = '/var/tmp'
# system logs to monitor
SYSTEM_LOGS = {'kern_log': '/var/log/kern.log',
               'messages_log': '/var/log/messages',
               'auth_log': '/var/log/auth.log',
               'syslog_log': '/var/log/syslog',
               'dmesg': '/var/log/dmesg',
               'sysstat': '/var/log/sysstat',
               }


class LogGather(object):
    def get_or_create_dump_dir(cls):
        '''
        creates a dump directory for log storage
        returns: location dump directory
        '''
        dir_date = time.strftime('%Y-%m-%d_%H-%M-%S')
        dir_name = os.path.dirname('{}/stricken_dump-{}/'.format(TMP_DIR, dir_date))
        if not os.path.exists(dir_name):
            os.makedirs(dir_name)
            log.debug('created dir: {}'.format(dir_name))
        return '{}/'.format(dir_name)

    def rm_dir(cls, dir_path):
        '''
        wraps shutil.rmtree with error handling
        '''
        try:
            log.debug('removing {}'.format(dir_path))
            rmtree(dir_path)
        except Exception as e:
            log.debug('unable to remove {} because {}'.format(dir_path, e))

    def copy(cls, origin, dest):
        '''
        determines if copying files or directories, and performs that action
        '''
        try:
            if os.path.isfile(origin):
                copy2(origin, dest)
            elif os.path.isdir(origin):
                dest_dir = origin.split('/')[-1]
                dest = os.path.join(dest, dest_dir)
                copytree(origin, dest)
            log.debug('copied {} to {}'.format(origin, dest))
        except Exception as e:
            log.error('unable to copy {} because {}'.format(origin, e))

    def clean_dump_dir(cls):
        '''
        cleans dump directories that are older than 30 days, or more than 5
        '''
        count = 0
        days_old = 86400*30
        now = time.time()
        time_delta = now - days_old
        directory = os.path.dirname(TMP_DIR)

        for root, dirs, files in os.walk(directory, topdown=True):
            dirs.sort(reverse=True)
            for d in dirs:
                dir_path = os.path.join(root, d)
                if 'stricken_dump' in d:
                    count += 1
                    dir_time = os.path.getmtime(dir_path)
                    # validate if only 5 dirs
                    if count > 5:
                        cls.rm_dir(dir_path)
                    # validate time
                    elif time_delta > dir_time:
                        cls.rm_dir(dir_path)

    def process_log(cls, log_origin, log_dest):
        '''
        processes a log file, backs up a copy of the log
        TBD: compress, copy multiple
        '''
        # copy file
        cls.copy(log_origin, log_dest)


class LogGatherScript(object):
    def __init__(cls):
        cls.lg = LogGather()

    def run(cls, args):
        '''
        default run function that processes all related components
        '''

        # set log debug if passed, otherwise default to warn
        if args.debug:
            log.setLevel(logging.DEBUG)
        else:
            log.setLevel(logging.WARNING)

        if args.action == 'start':
            # create temp dump directory for log stash
            dump_dir = cls.lg.get_or_create_dump_dir()

            # process logs
            for log_name, log_file in SYSTEM_LOGS.iteritems():
                cls.lg.process_log(log_file, dump_dir)

            # clean up after myself
            cls.lg.clean_dump_dir()

    def _add_arguments(cls, parser):
        parser.add_argument(
            'action',
            help='start/stop action')
        parser.add_argument(
            '-d', '--debug',
            help='enable debug messages',
            action='store_true')


def main():
    cmd = LogGatherScript()

    # add arguments
    parser = argparse.ArgumentParser()
    cmd._add_arguments(parser)
    args = parser.parse_args()

    return(cmd.run(args))


if __name__ == '__main__':
    exit(main())
