# twitch_svclogs::logger instance type
define twitch_svclogs::logger(
  $ensure          = present,
  $env             = pick($::twitch_environment, $twitch_environment, 'production'),
  $daemon_dir      = '/var/lib/service',
  $nice_value      = 4,
  $log_group_name  = undef,
  $log_stream_name = "${::pop}/${::hostname}",
  $datetime_format = '%b %d %H:%M:%S',
  $buffer_duration = 5000, # ms
  $batch_size      = 32768, # bytes
  $batch_count     = 1000,
  $owner           = 'video',
  $account         = 'twitch-video',
  $region          = 'us-west-2',
  $runas_user      = 'svclogs',

) {

  validate_re($ensure, '^(present|absent)$')
  validate_re($env, '^(development|staging|production)$')
  validate_re($account, '^(twitch-dumbodrop|twitch-video|twitch-video-ops|instance-role)$')
  validate_re($region, '^(us-west-2|us-east-1)$')
  validate_string($daemon_dir, $log_stream_name, $datetime_format)
  validate_integer($nice_value)
  validate_integer($buffer_duration)
  validate_integer($batch_size)
  validate_integer($batch_count)

  include ::twitch_svclogs

  $base_dir    = "${twitch_svclogs::base_dir}-${name}"
  $config_logs = $twitch_svclogs::config_logs
  $config_auth = $twitch_svclogs::config_auth

  $config_dir = "${base_dir}/conf.d"
  $logsvc_dir = "${daemon_dir}/${name}/log"

  $ensure_dir = $ensure ? {
    present => directory,
    absent  => absent,
  }

  file { $base_dir:
    ensure => $ensure_dir,
    owner  => 'root',
    group  => 'root',
    mode   => '0755',
  }

  file { "${base_dir}/${config_logs}":
    ensure => $ensure,
    owner  => 'root',
    group  => 'root',
    mode   => '0644',
    source => "puppet:///modules/${module_name}/logging.conf",
  }

  if ( $account == 'instance-role' ) {
    file { "${base_dir}/${config_auth}":
      ensure  => $ensure,
      owner   => 'root',
      group   => 'root',
      mode    => '0644',
      content => template("${module_name}/awscli.erb"),
      before  => File["${logsvc_dir}/run"],
    }
  } else {
    twitch_sandstorm_agent::template { $name:
      ensure      => $ensure,
      destination => "${base_dir}/${config_auth}",
      contents    => template("${module_name}/awscli.erb"),
      owner       => 'root',
      group       => $runas_user,
      mode        => '0640',
      before      => File["${logsvc_dir}/run"],
      require     => File[$base_dir],
    }
  }

  if ($log_group_name) {
    validate_string($log_group_name)
    $_log_group_name = $log_group_name
  } else {
    $_log_group_name = "svclogs/${name}"
  }

  # lets keep names consistent in template
  $_log_stream_name = $log_stream_name

  file { $logsvc_dir:
    ensure => $ensure_dir,
    owner  => 'root',
    group  => 'root',
    mode   => '0755',
  }

  file { "${logsvc_dir}/run":
    ensure  => $ensure,
    owner   => 'root',
    group   => 'root',
    mode    => '0755',
    content => template("${module_name}/logger.erb"),
  }

}
