# twitch_svclogs::service::instance instance type
define twitch_svclogs::service::instance (
    $ensure             = 'present',
    $env                = pick($::twitch_environment, $twitch_environment, 'production'),
    $account            = 'twitch-video',
    $owner              = 'video',
    $region             = 'us-west-2',
    $nice_value         = 4,
    $use_milliner       = false,
    $use_default_region = false,
) {

  include ::twitch_svclogs
  $base_dir    = $twitch_svclogs::base_dir
  $config_file = $twitch_svclogs::config_file
  $config_logs = $twitch_svclogs::config_logs
  $config_auth = $twitch_svclogs::config_auth
  $runas_user  = $twitch_svclogs::runas_user

  validate_re($ensure, '^(present|absent)$')
  validate_string($runas_user)
  validate_integer($nice_value)
  validate_bool(str2bool($use_milliner))

  $statefile    = "/var/tmp/${name}.state"
  $instance_dir = "${base_dir}/${name}"
  $config_dir   = "${instance_dir}/conf.d"

  $ensure_dir = $ensure ? {
    present => directory,
    absent  => absent,
  }

  file { $instance_dir:
    ensure  => $ensure_dir,
    owner   => 'root',
    group   => 'root',
    mode    => '0755',
    require => File[$base_dir],
  }
  # this conditional is here to allow for puppet to smoothly delete configuration directories of logfile instances.
  # These keys are not specified in the include case to minimize configuration management mishaps that may occur
  # by forcing these parameters in after they've already been configured on existing machines.
  if $ensure == 'absent' {
    File[$instance_dir] {
      recurse => true,
      force   => true,
    }
  }

  file { $config_dir:
    ensure  => $ensure_dir,
    owner   => 'root',
    group   => 'root',
    mode    => '0755',
    require => File[$instance_dir],
  }

  file { "${instance_dir}/${config_file}":
    ensure  => $ensure,
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    content => template("${module_name}/svclogs.erb"),
    require => File[$config_dir],
  }

  file { "${instance_dir}/${config_logs}":
    ensure  => $ensure,
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    source  => "puppet:///modules/${module_name}/logging.conf",
    require => File[$config_dir],
  }

  if ( $account == 'instance-role' ) {
    file { "${instance_dir}/${config_auth}":
      ensure  => $ensure,
      owner   => 'root',
      group   => 'root',
      mode    => '0644',
      content => template("${module_name}/awscli.erb"),
      notify  => Daemontools::Supervise[$name],
      before  => Daemontools::Supervise[$name],
      require => [ File[$instance_dir], User[$runas_user], ],
    }
  } else {
    twitch_sandstorm_agent::template { $name:
      ensure      => $ensure,
      destination => "${instance_dir}/${config_auth}",
      contents    => template("${module_name}/awscli.erb"),
      owner       => 'root',
      group       => $runas_user,
      mode        => '0640',
      notify      => Daemontools::Supervise[$name],
      before      => Daemontools::Supervise[$name],
      require     => [ File[$instance_dir], User[$runas_user], ],
    }
  }

  if str2bool($use_milliner) {
    $predaemon = [
      'test -f /etc/profile.d/proxy.sh && source /etc/profile.d/proxy.sh',
      'export PATH=/usr/local/bin:/usr/bin:/bin',
      "export AWS_CONFIG_FILE=${instance_dir}/${config_auth}",
      'export AWS_STS_REGIONAL_ENDPOINTS=regional',
      'export AWS_CSM_ENABLED=true',
      'export AWS_CSM_CLIENT_ID=SvcLogs',
      "test -f ${statefile} && chown ${runas_user} ${statefile}",
      'systemctl is-active -q video-milliner && test -f /etc/milliner-client-env.sh || (echo milliner not ready && sleep 5 && exit 1)',
      'source /etc/milliner-client-env.sh',
      ]
  } else {
    $predaemon = [
      'test -f /etc/profile.d/proxy.sh && source /etc/profile.d/proxy.sh',
      'export PATH=/usr/local/bin:/usr/bin:/bin',
      "export AWS_CONFIG_FILE=${instance_dir}/${config_auth}",
      "test -f ${statefile} && chown ${runas_user} ${statefile}",
      ]
  }


  $daemon_cmd = join([
    "nice -n ${nice_value} aws logs push",
        "--config-file ${instance_dir}/${config_file}",
        "--additional-configs-dir ${config_dir}",
    ], ' ')

  daemontools::supervise { $name:
    ensure         => $ensure,
    env            => $env,
    manage_service => true,
    wd             => $instance_dir,
    user           => $runas_user,
    predaemon      => $predaemon,
    daemon         => "${daemon_cmd} 2>&1",
    syslog         => 'local3',
    daemon_dir     => '/var/lib/service',
    require        => File[$instance_dir, $config_dir,
                        "${instance_dir}/${config_file}",
                        "${instance_dir}/${config_logs}"],
  }
}
