# Twitch.tv wrapper around saz/rsyslog OSS module
class twitch_syslog (
  $rsyslog_run_user                         = $twitch_syslog::params::rsyslog_run_user,
  $rsyslog_run_group                        = $twitch_syslog::params::rsyslog_run_group,
  $rsyslog_log_user                         = $twitch_syslog::params::rsyslog_log_user,
  $rsyslog_log_group                        = $twitch_syslog::params::rsyslog_log_group,
  $rsyslog_perm_file                        = $twitch_syslog::params::rsyslog_perm_file,
  $rsyslog_perm_dir                         = $twitch_syslog::params::rsyslog_perm_dir,
  $rsyslog_escapecontrolcharactersonreceive = $twitch_syslog::params::rsyslog_escapecontrolcharactersonreceive,
  $rsyslog_repeatedmsgreduction             = $twitch_syslog::params::rsyslog_repeatedmsgreduction,
  $rsyslog_umask                            = $twitch_syslog::params::rsyslog_umask,
  $rsyslog_udpserveraddress                 = $twitch_syslog::params::rsyslog_udpserveraddress,
  $rsyslog_udpserverport                     = $twitch_syslog::params::rsyslog_udpserverport,
  $rsyslog_modules                          = $twitch_syslog::params::rsyslog_modules,
  $rsyslog_rotate                           = $twitch_syslog::params::rsyslog_rotate,
  $rsyslog_relp_package_name                = $twitch_syslog::params::relp_package_name,
  $rsyslog_gnutls_package_name              = $twitch_syslog::params::gnutls_package_name,
  $rsyslog_reopen_logs_command              = $twitch_syslog::params::rsyslog_reopen_logs_command,
  $rsyslog_omit_local_logging               = $twitch_syslog::params::rsyslog_omit_local_logging,
  $rsyslog_package_status                   = $twitch_syslog::params::rsyslog_package_status,
  $rsyslog_local_loglevel                   = $twitch_syslog::params::rsyslog_local_loglevel,
  $rsyslog_max_message_size                 = $twitch_syslog::params::rsyslog_max_message_size,
  $rsyslog_imjournalrun                     = $twitch_syslog::params::rsyslog_imjournalrun,
  $rsyslog_purge_rsyslogd                   = $twitch_syslog::params::rsyslog_purge_rsyslogd,
  $syslog_retention_days                    = $twitch_syslog::params::syslog_retention_days,
  $log_archive_purge_days                   = $twitch_syslog::params::log_archive_purge_days,
  # If your team has a centralized logging cluster and you'd like to ship
  # local3 logs to it, set this to the hostname of the cluster.
  $local3_ingest_server                     = $twitch_syslog::params::local3_ingest_server,
  $var_log_group                            = $twitch_syslog::params::var_log_group,
  $rsyslog_system_log_rate_limit_interval   = $twitch_syslog::params::rsyslog_system_log_rate_limit_interval,
  $rsyslog_system_log_rate_limit_burst      = $twitch_syslog::params::rsyslog_system_log_rate_limit_burst,
) inherits twitch_syslog::params {

  validate_string($rsyslog_reopen_logs_command)
  validate_string($rsyslog_run_user)
  validate_string($rsyslog_run_group)
  validate_string($rsyslog_log_user)
  validate_string($rsyslog_log_group)
  validate_string($rsyslog_perm_file)
  validate_string($rsyslog_perm_dir)
  validate_string($rsyslog_escapecontrolcharactersonreceive)
  validate_string($rsyslog_repeatedmsgreduction)
  validate_string($rsyslog_umask)
  validate_string($var_log_group)
  validate_array($rsyslog_modules)
  validate_bool($rsyslog_omit_local_logging)
  validate_string($rsyslog_package_status)
  validate_integer($rsyslog_local_loglevel)
  validate_integer($log_archive_purge_days, 120, 1)

  if $rsyslog_udpserveraddress {
    validate_ip_address($rsyslog_udpserveraddress)
  }
  if $rsyslog_udpserverport {
    validate_integer($rsyslog_udpserverport)
  }

  $imuxsock_module = [
    '# provides support for local system logging',
    "module(load=\"imuxsock\"
      SysSock.RateLimit.Interval=\"${rsyslog_system_log_rate_limit_interval}\"
      SysSock.RateLimit.Burst=\"${rsyslog_system_log_rate_limit_burst}\"
      SysSock.UsePIDFromSystem=\"on\"
    )",
  ]

  $modules = concat($rsyslog_modules,$imuxsock_module)

  class{ '::rsyslog':
    run_user            => $rsyslog_run_user,
    run_group           => $rsyslog_run_group,
    log_user            => $rsyslog_log_user,
    log_group           => $rsyslog_log_group,
    perm_file           => $rsyslog_perm_file,
    perm_dir            => $rsyslog_perm_dir,
    modules             => $modules,
    relp_package_name   => $rsyslog_relp_package_name,
    gnutls_package_name => $rsyslog_gnutls_package_name,
    omit_local_logging  => $rsyslog_omit_local_logging,
    package_status      => $rsyslog_package_status,
    max_message_size    => $rsyslog_max_message_size,
    purge_rsyslog_d     => $rsyslog_purge_rsyslogd,
  }

  # Remote configuration is handled later, we want it last in the chain.
  class { '::rsyslog::client':
    log_remote => false,
  }

  # Override options 
  twitch_systemd::drop_in { 'rsyslog-overrides.conf':
    unit    => 'rsyslog.service',
    content => template("${module_name}/rsyslog.systemd.dropin.erb"),
    require => Class['::rsyslog'],
  }

  $_rsyslog_imjournal_ensure = $rsyslog_imjournalrun ? {
    true  => 'present',
    false => 'absent',
  }

  class { '::twitch_syslog::journald':
    ensure => $_rsyslog_imjournal_ensure,
  }

  if ($rsyslog_escapecontrolcharactersonreceive) {
    ::rsyslog::snippet { 'EscapeControlCharactersOnReceive':
      content => "\$EscapeControlCharactersOnReceive ${rsyslog_escapecontrolcharactersonreceive}",
    }
  }

  ::rsyslog::snippet { 'RepeatedMsgReduction':
    content => "\$RepeatedMsgReduction ${rsyslog_repeatedmsgreduction}",
  }

  ::rsyslog::snippet { 'Umask':
    content => "\$Umask ${rsyslog_umask}",
  }

  ::rsyslog::snippet { 'UDPServerAddress':
    ensure => 'absent',
  }

  ::rsyslog::snippet { 'UDPServerRun':
    ensure => 'absent',
  }

  ::rsyslog::snippet { 'imudp-config':
    content => "input(type=\"imudp\" address=\"${rsyslog_udpserveraddress}\" port=\"${rsyslog_udpserverport}\")",
  }

  file { '/var/log':
    ensure  => directory,
    owner   => 'root',
    group   => $var_log_group,
    mode    => '0775',
    require => Class['::rsyslog::install'],
  }

  file { '/var/log/jtv':
    ensure  => directory,
    owner   => 'syslog',
    group   => 'adm',
    mode    => '0755',
    require => Class['::rsyslog'],
  }

  file { '/var/log/archive':
    ensure  => directory,
    owner   => 'root',
    group   => 'root',
    mode    => '0755',
    require => Class['::rsyslog'],
  }

  file {'/usr/local/bin/rsyslog_reopen_logs.sh':
    ensure  => present,
    owner   => 'root',
    group   => 'root',
    mode    => '0755',
    content => "#!/bin/sh\n${rsyslog_reopen_logs_command} > /dev/null\n",
  }

  cron::job { 'purge_orphaned_log':
    ensure  => present,
    command => "find /var/log/archive -type f -mtime +${log_archive_purge_days} -delete",
    user    => root,
    hour    => fqdn_rand(23, 'purge_orphaned_hour'),
    minute  => fqdn_rand(59, 'purge_orphaned_minute'),
    require => Class['::cron::install'],
  }

  ::rsyslog::snippet { '30-preserve-fqdn':
    content => "\$PreserveFQDN on",
  }

  ::rsyslog::snippet { '41-jtv-local3-logs':
    content => template("${module_name}/41-jtv-local3-logs.conf.erb"),
  }
  ::rsyslog::snippet{ '42-jtv-local4-logs':
    content => template("${module_name}/42-jtv-local4-logs.conf.erb"),
  }
  ::rsyslog::snippet{ '44-jtv-local5-json':
    content => template("${module_name}/44-jtv-local5-json.conf.erb"),
  }
  ::rsyslog::snippet { '49-jtv':
    content => template("${module_name}/49-jtv.conf.erb"),
  }
  ::rsyslog::snippet { '50-default':
    content => template("${module_name}/50-default.conf.erb"),
  }

  logrotate::rule { 'jtv_log_dir':
    ensure          => present,
    path            => '/var/log/jtv/*',
    olddir          => '/var/log/archive',
    missingok       => true,
    rotate          => $rsyslog_rotate,
    rotate_every    => 'day',
    compress        => true,
    compresscmd     => '/usr/bin/pbzip2',
    compressoptions => '-1p4',
    compressext     => '.bz2',
    sharedscripts   => true,
    create          => false,
    postrotate      => '/usr/local/bin/rsyslog_reopen_logs.sh',
    size            => '0',
    require         => File['/var/log/archive'],
  }

  logrotate::rule { 'jtv':
    ensure       => present,
    path         => '/var/log/jtv.log',
    olddir       => '/var/log/archive',
    missingok    => true,
    rotate       => $rsyslog_rotate,
    rotate_every => 'day',
    create       => true, #644 syslog adm',
    create_mode  => '0644',
    create_owner => 'syslog',
    create_group => 'adm',
    postrotate   => '/usr/local/bin/rsyslog_reopen_logs.sh',
  }

  logrotate::rule { 'syslog':
    ensure       => present,
    path         => '/var/log/syslog',
    olddir       => '/var/log/archive',
    missingok    => true,
    rotate       => $syslog_retention_days,
    rotate_every => 'day',
    postrotate   => '/usr/local/bin/rsyslog_reopen_logs.sh',
    compress     => true,
  }

  logrotate::rule { 'mail':
    ensure       => present,
    path         => [
      '/var/log/mail.info',
      '/var/log/mail.warn',
      '/var/log/mail.err',
      '/var/log/mail.log',
    ],
    olddir       => '/var/log/archive',
    missingok    => true,
    rotate       => 7,
    rotate_every => 'day',
    create       => false,
    postrotate   => '/usr/local/bin/rsyslog_reopen_logs.sh',
  }

  logrotate::rule { 'rsyslog':
    ensure        => present,
    path          => [
      '/var/log/daemon.log',
      '/var/log/kern.log',
      '/var/log/user.log',
      '/var/log/rsyslog.log',
      '/var/log/debug',
      '/var/log/messages',
    ],
    olddir        => '/var/log/archive',
    missingok     => true,
    rotate        => 4,
    rotate_every  => 'week',
    compress      => true,
    sharedscripts => true,
    ifempty       => false,
    postrotate    => '/usr/local/bin/rsyslog_reopen_logs.sh',
  }

  logrotate::rule { 'authlog':
    ensure        => present,
    path          => [
      '/var/log/auth.log',
    ],
    olddir        => '/var/log/archive',
    missingok     => true,
    rotate        => 30,
    size          => '1M',
    compress      => true,
    delaycompress => true,
    sharedscripts => true,
    ifempty       => false,
    postrotate    => '/usr/local/bin/rsyslog_reopen_logs.sh',
  }

  logrotate::rule { 'wtmp':
    ensure        => 'present',
    missingok     => true,
    path          => '/var/log/wtmp',
    olddir        => '/var/log/archive',
    copy          => true,
    create        => true,
    create_mode   => '0664',
    create_owner  => 'root',
    create_group  => 'utmp',
    rotate_every  => 'month',
    rotate       => 1,
  }

  logrotate::rule { 'btmp':
    ensure       => 'present',
    missingok    => true,
    path         => '/var/log/btmp',
    olddir       => '/var/log/archive',
    copy         => true,
    create       => true,
    create_mode  => '0660',
    create_owner => 'root',
    create_group => 'utmp',
    rotate_every => 'month',
    rotate       => 1,
  }

}
