# class twitch_syslog::journald
# controls systemd's journald configuration
#
# @param ensure              [String]
# @param rate_limit_interval [String] Configures the rate limiting that is applied to all messages generated on the system.
# @param rate_limit_burst    [String] Combined with rate_limit_interval, this rate limiting is applied per-service.
#                                     Time units may be specified in the following units: "s", "min", "h", "ms", "us"
# @param max_retention_sec   [String] The maximum time to store journal entries.
#                                     The units "year", "month", "week", "day", "h" or " m" override the default unit of seconds.
# @param forward_to_syslog   [Bool]   Forward messages received by the journal daemon to a traditional syslog daemon
#
class twitch_syslog::journald(
  $ensure              = 'present',
  $rate_limit_interval = '0',
  $rate_limit_burst    = '0',
  $max_retention_sec   = undef,
  $forward_to_syslog   = true,
) {
  validate_string($ensure)
  validate_string($rate_limit_interval)
  validate_string($rate_limit_burst)
  validate_bool($forward_to_syslog)

  # write journald's config file.
  twitch_systemd::drop_in { 'journald.conf':
    ensure  => $ensure,
    unit    => 'journald.conf',
    path    => '/etc/systemd',
    content => template("${module_name}/journald.conf.erb"),
    notify  => Class['systemd'],
  }

  exec { 'rm /etc/rsyslog.d/00-journald-config.conf':
    onlyif => 'test -f /etc/rsyslog.d/00-journald-config.conf',
    before => Class['rsyslog::install'],
  }
}
