# === Define: twitch_systemd::drop_in
#
#   Places a drop in unit file in the correct location.
#
# === Parameters
#
# [*unit*] (string)
#   Name of the unit
#
# [*ensure*] (file/absent)
#   Create unit file, or delete unit file.
#
# [*path*] (string)
#   Where the unit file will be placed. Default: /etc/systemd/system
#   Another valid option may be /lib/systemd/system
#   The default comes from systemd::service define.
#
# [*content*] (string)
#   Must provide content OR source for unit file contents.
#   Content should be data, such as from template().
#
# [*source*] (string)
#   Must provide content OR source for unit file contents.
#   Source should probably be a puppet:/// path.
#
#
# [*manage*] (boolean)
#   Remove non-puppeted resources from the drop-in directory.
#
define twitch_systemd::drop_in (
  $unit,
  $ensure  = present,
  $path    = '/etc/systemd/system',
  $content = undef,
  $source  = undef,
  $manage  = true,
) {
  include ::systemd

  validate_re($ensure, '^(present|absent)$')
  validate_re($name, '\.(service|slice|socket|timer|mount|target|path|automount|device|scope|swap|conf)$')
  validate_absolute_path($path)

  $ensure_drop_in_dir = $ensure ? {
    present => directory,
    default => absent
  }

  file { "${path}/${unit}.d":
    ensure  => $ensure_drop_in_dir,
    recurse => $manage,
    owner   => 'root',
    group   => 'root',
    purge   => $manage,
    force   => true,
    mode    => '0755',
  }

  file { "${path}/${unit}.d/${name}":
    ensure  => $ensure,
    content => $content,
    source  => $source,
    owner   => 'root',
    group   => 'root',
    mode    => '0444',
    notify  => Class['::systemd'],
  }
}
