# == Define: twitch_systemd::service
#
# Creates a systemd service with a templated unit file.
# The idea is to add an easy transition for daemontools-using manifests.
#
# === Parameters
#
# [*cwd*] (string)
#   Working directory to run the command in.
#
# [*command*] (string)
#   Binary/Script to run. This is run by ExecStart.
#
# [*options*] (string)
#   Options and parameters to pass to the command. This is optional and params
#   may also be passed in 'command'. This is just an appended string.
#
# [*stopcmd*] (string)
#   OPTIONAL: Command to run to stop this service from running. Default is
#   to use systemd's control: SIGKILL/SIGTERM (default is usually fine)
#
# [*reload*] (string)
#   OPTIONAL: By default sends a HUP. If you don't want that,
#   set this to '' (empty string).
#   Generally safe because reload is not a commonly performed task.
#
# [*restart_on_change*] (boolean)
#   OPTIONAL: By default if the systemd unit file is changed, we will
#   restart the service to pickup any command line options that may
#   have changed.
#   Set to false to disable this behavior
#
# [*user*] (string)
#   User to run this service as. Default is $name.
#   Most services will need to change this.
#
# [*group*] (string)
#   Group to run this service as. Default is $user.
#   Most services will need to change this.
#
# [*logname*] (string)
#   OPTIONAL: Name to log to syslog as. Default is $name.
#
# [*pidfile*] (string/path)
#   OPTIONAL: If your service forks, it should create a pid file.
#   Specify that path here. If the service does not fork, don't set this.
#
# [*timeout*] (interger/seconds or string duration: '5m30s')
#   OPTIONAL: Configures the time to wait for start-up and stop.
#   Generally used to specify time between kill signals.
#
# [*noproxy*] (boolean)
#   OPTIONAL: Defaults to false. This removes sourcing of the proxy file.
#   You probably don't need to change this. Most apps do the right thing.
#
# [*envvars*] (list)
#   OPTIONAL: You may provide a list of environment variables to set
#   before executing the service.
#
# [*envfile*] (list)
#   OPTIONAL: You may provide a list of environment files to source
#   before executing the service.
#
# [*after*] (list)
#   OPTIONAL: Defaults to ['network.target'] which is what most apps need.
#   Systemd tries to bring this unit up after whatever you define here.
#
# [*requires*] (list)
#   OPTIONAL: Defaults to $after which is what most apps need.
#   Systemd requires these units to be up before bringing this one up.
#
# [*privtmp*] (boolean)
#   OPTIONAL: Defaults to false. Systems provides a PrivateTMP feature.
#
# [*nofile*] (string)
#   OPTIONAL: Limit number of open files.
#
# [*core*] (string)
#   OPTIONAL: Limit size of core dumps.
#
# [*memlock*] (string)
#   OPTIONAL: Limit size of locked memory.
#
# [*killmode*] (string)
#   OPTIONAL: Set the killmode for the service in systemd.
#
# [*before_u*] (list)
#   OPTIONAL: The units listed in this parameter will not be started
#   until the current unit is marked as started.
#
# [*wants*] (list)
#   OPTIONAL: Systemd will attempt to start any units listed
#   here when this unit is activated.
#
#  Example:
#  -----------------------------------------------------------------
#  twitch_systemd::service { 'nrdp':
#    command => '/usr/local/nrdp/twitch_nrdp.py',
#    options => '--auto --monitor http://central-nagios.justin.tv/',
#    cwd     => '/usr/local/nrdp/',
#  }
#
#  That creates:
#
# [Unit]
# Description = Twitch nrdp Service
# After=network.service
#
# [Service]
# ExecStart=/usr/local/nrdp/twitch_nrdp.py --auto --monitor http://central-nagios.justin.tv/
# ExecReload=/bin/kill -HUP $MAINPID
# Restart=always
# StandardOutput=syslog
# StandardError=syslog
# SyslogIdentifier=nrdp
# User=nrdp
# Group=nrdp
# Type=simple
# EnvironmentFile=-/etc/profile.d/proxy.sh
# EnvironmentFile=-/etc/default/twitch_nrdp
# PermissionsStartOnly=false
# RestartSec=500ms
# PrivateTmp=no
# WorkingDirectory=/usr/local/nrdp/
#
# [Install]
# WantedBy=multi-user.target
#
#
define twitch_systemd::service (
  $command,
  $enable            = true,
  $cwd               = '/tmp',
  $options           = '',
  $reload            = '/bin/kill -HUP $MAINPID',
  $execpre           = undef,
  $stopcmd           = undef,
  $user              = $name,
  $group             = pick($user,$name),
  $logname           = $name,
  $pidfile           = undef,
  $timeout           = 0,
  $noproxy           = false,
  $envvars           = [],
  $envfile           = [],
  $after             = ['network.target'],
  $requires          = pick($after,['network.target']),
  $privtmp           = false,
  $nofile            = undef,
  $core              = undef,
  $memlock           = undef,
  $killmode          = undef,
  $before_u          = [],
  $wants             = [],
  $restart_on_change = true,
  $standard_output   = 'syslog',
  $standard_error    = 'syslog',
  $syslog_identifier = $name,
  $restart           = undef,
) {
  include twitch_systemd
  # If pidfile is provided, set type to forking.
  $srvctype = $pidfile ? {
    undef   => 'simple',
    default => 'forking',
  }
  $envfiles = $noproxy ? {
    false   => concat(['-/etc/proxy.d/proxy.sh'], $envfile),
    default => $envfile,
  }
  $ensure   = $enable ? {
    true    => 'running',
    default => 'stopped'
  }
  $_reload  = $reload ? {
    ''      => undef,
    default => $reload,
  }
  $_timeout = $timeout ? {
    undef   => undef,
    ''      => undef,
    0       => undef,
    default => $timeout,
  }
  $_group = $group ? {
    'nobody' => 'nogroup',
    default  => $group,
  }

  # Create a place for temporary service items; like a socket and pid file.
  $_execpre = $execpre ? {
    undef   => [
      "/bin/mkdir -p /run/twitch_${name}",
      "/bin/chown ${user}:${_group} /run/twitch_${name}",
    ],
    default => $execpre,
  }

  $bonusenv = "-${::core::default_dir}/twitch_${name}"


  validate_string($command,$options,$reload,$stopcmd,$user,$group)
  validate_string($cwd,$logname,$pidfile,$killmode,$core,$memlock)
  validate_array($envvars,$envfile,$before_u,$wants,$after)
  validate_bool($noproxy,$enable,$privtmp)

  ::systemd::service { $name:
    description            => "Twitch ${name} Service",
    servicename            => $logname,
    execstart              => "${command} ${options}",
    execstartpre           => $_execpre,
    execstop               => $stopcmd,
    execreload             => $_reload,
    user                   => $user,
    group                  => $_group,
    type                   => $srvctype,
    env_vars               => suffix(prefix($envvars, '"'), '"'),
    environment_files      => concat($envfiles, $bonusenv),
    permissions_start_only => true,
    working_directory      => $cwd,
    pid_file               => $pidfile,
    restart_sec            => '500ms',
    after_units            => $after,
    requires               => $requires,
    timeoutsec             => $_timeout,
    private_tmp            => $privtmp,
    limit_nofile           => $nofile,
#    limit_core             => $core,
#    limit_memlock          => $memlock,
    killmode               => $killmode,
#    syslog_facility        => 'local3',
    before_units           => $before_u,
    wants                  => $wants,
    standard_output        => $standard_output,
    standard_error         => $standard_error,
    syslog_identifier      => $syslog_identifier,
    restart                => $restart,
  }

  service { $name:
    ensure     => $ensure,
    enable     => $enable,
    hasrestart => true,
    hasstatus  => true,
    require    => Exec['systemctl reload'],
  }

  if $restart_on_change {
    Service[$name] {
      subscribe => Systemd::Service[$name]
    }
  }

}
