# Class: twitch_transcode::vp9::haproxy
#
# Sets up the haproxy service needed by vp9 transcode

class twitch_transcode::vp9::haproxy (
  $listen_port     = 8080,
  $origin_port     = 9090,
  $stats_port      = 2017,
  $instname        = 'transcode_vp9',
  $default_backend = 'video-origin',
  $maxconn         = 4000,
  $nbproc          = 1,
  $listen_address  = '0.0.0.0',
) inherits ::twitch_transcode::vp9::params {

  include twitch_vars

  $allow_ips = concat($::twitch_vars::twitch_networks, '127.0.0.0/8')
  $ip_whitelist_acl_rules = prefix($allow_ips, 'whitelisted_ip src ')

  ## should export a lot of this to hiera/params
  $global_options = {
    'user'         => 'www-data',
    'group'        => 'www-data',
    'daemon'       => '',
    'quiet'        => '',
    'pidfile'      => "/var/run/haproxy-${instname}.pid",
    'maxconn'      => $maxconn,
    'chroot'       => '/var/chroot/haproxy',
    'nbproc'       => $nbproc,
    'log-tag'      => "haproxy_${instname}",
    'log'          => '127.0.0.1 local3',
    'stats'        => "socket /var/run/haproxy-${instname}.stat user root group root mode 666 level operator",
  }

  $defaults_options = {
    'log'     => 'global',
    'balance' => 'roundrobin',
    'option'  => [
      'redispatch',
      'httplog',
    ],
    'timeout' => [
      'queue 500',
    ]
  }

  $frontend_options = {
    'default_backend' => $default_backend,
    'mode'            => 'http',
    'timeout'         => [
      'http-request    1000',
      'http-keep-alive 10000',
    ],
    'acl'            => $ip_whitelist_acl_rules,
    'http-request' =>  [
      'deny if !whitelisted_ip',
    ],
  }

  $backend_options = {
    'mode'    => 'http',
    'option'  => [
      'httpchk GET /debug/health',
      'forwardfor',
    ],
    'timeout' => ['http-keep-alive 1000'],
  }

  # sets up basic haproxy instance
  twitch_haproxy::instance { $instname:
    defaults_options => $defaults_options,
    global_options   => $global_options,
    stats_port       => $stats_port,
  }

  # sets up haproxy frontend for vp9
  twitch_haproxy::frontend{ "${instname}-in":
    instance_name => $instname,
    ports         => [$listen_port],
    ipaddress     => $listen_address,
    options       => $frontend_options,
  }

  # sets up video origin haproxy backend
  class {'twitch::haproxy::backends::video_origin':
    haproxy_instance => $instname,
    port             => $origin_port,
  }
  ##  twitch_haproxy::backend { $default_backend:
  ##    instance_name       => $instname,
  ##    server_list         => [
  ##      "localhost:${origin_port} fall 1 rise 2",
  ##    ],
  ##    default_remote_port => $origin_port,
  ##    server_options      => 'check inter 1000',
  ##    options             => $backend_options,
  ##  }
}
