# Twitch specific unbound config module.
class twitch_unbound (
  $role                  = $twitch_unbound::params::role,
  $access_cidrs          = $twitch_unbound::params::access_cidrs,
  $interface             = $twitch_unbound::params::interface,
  $interface_automatic   = $twitch_unbound::params::interface_automatic,
  $remote_interface      = $twitch_unbound::params::remote_interface,
  $val_permissive_mode   = $twitch_unbound::params::val_permissive_mode,
  $val_log_level         = $twitch_unbound::params::val_log_level,
  $num_threads           = $twitch_unbound::params::num_threads,
  $statistics_interval   = $twitch_unbound::params::statistics_interval,
  $statistics_cumulative = $twitch_unbound::params::statistics_cumulative,
  $vpc_resolver_domains  = $twitch_unbound::params::vpc_resolver_domains,
  $package_version       = $twitch_unbound::params::package_version,
  $package_name          = $twitch_unbound::params::package_name,
  $env                   = $twitch_unbound::params::env,
  $custom_server_conf    = [],
  $forwarders            = {},
  $enable_prefetch       = $twitch_unbound::params::enable_prefetch,
) inherits twitch_unbound::params {
  $envfile               = $twitch_unbound::params::envfile

  include twitch_unbound::logging
  include twitch_vars

  $_access_cidrs = concat($::twitch_vars::twitch_networks, $access_cidrs)

  twitch_diamond::collector { 'TwUnboundCollector':
    enable  => true,
    source  => true,
    options => {
      histogram => true,
    },
  }

  ensure_resource('twitch_diamond::collector', 'UDPCollector', {
    enable  => true,
    options => {
      allowed_names => '',
    },
  })

  package { $package_name:
    ensure => $package_version,
  }

  unless $package_version =~ /installed|present|absent|purged|held|latest/ {
    apt::pin { $package_name:
      ensure   => present,
      packages => $package_name,
      priority => 600,
      version  => $package_version,
    }
  }
  # The default interface in the unbound::remote module is ['127.0.0.1', '::1']
  # Not all instances have an IPV6 localhost address. This is the workaround.
  class { '::unbound::remote':
    enable            => true,
    interface         => $remote_interface,
    server_key_file   => '/etc/unbound/unbound_server.key',
    server_cert_file  => '/etc/unbound/unbound_server.pem',
    control_key_file  => '/etc/unbound/unbound_control.key',
    control_cert_file => '/etc/unbound/unbound_control.pem',
    group             => 'unbound',
    confdir           => '/etc/unbound',
    require           => Package[$package_name],
  }

  # XXX: the slab options need to be powers of 2, we calculate the lowest closest power of 2
  # from the number of threads.
  $cache_slabs = clamp(16, unbound_calculate_slabs($num_threads), 64)

  class { '::unbound':
    interface             => $interface,
    access                => $_access_cidrs,
    # The anchor fetch command exec expects return code 1 as considered successful
    anchor_fetch_command  => '/bin/cp -fp /usr/share/dns/root.key /var/lib/unbound/root.key && exit 1 || exit 2',
    conf_d                => $twitch_unbound::params::unbound_confd_dir,
    statistics_interval   => $statistics_interval,
    extended_statistics   => 'yes',
    control_enable        => true,
    statistics_cumulative => $statistics_cumulative,
    interface_automatic   => $interface_automatic,
    do_ip6                => false,
    hide_identity         => true,
    hide_version          => true,
    verbosity             => 1,
    num_threads           => $num_threads,
    msg_cache_slabs       => $cache_slabs,
    rrset_cache_slabs     => $cache_slabs,
    infra_cache_slabs     => $cache_slabs,
    key_cache_slabs       => $cache_slabs,
    rrset_cache_size      => '2g',
    msg_cache_size        => '1g',
    so_rcvbuf             => '16m',
    custom_server_conf    => concat($twitch_unbound::params::default_custom_server_conf,$custom_server_conf),
    custom_conf           => $twitch_unbound::params::custom_conf,
    val_permissive_mode   => $val_permissive_mode,
    val_log_level         => $val_log_level,
    package_name          => false,
    prefetch              => $enable_prefetch,
    require               => Package[$package_name],
  }

  # this ensures that the package-helper script which comes in the ubuntu
  # package ignores trying to grab DNSSEC trust anchors as this doesn't
  # work without NAT on privately IP'ed machines.
  file { $envfile:
    ensure  => 'present',
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    content => "ROOT_TRUST_ANCHOR_UPDATE=false\n",
    before  => Class['::unbound'],
  }

  if str2bool($::systemd) {
    # This places a unit file before the package is installed.
    twitch_systemd::unit_file { 'unbound.service':
      ensure  => 'absent',
    }
    twitch_systemd::drop_in { 'twitch-overrides.conf':
      unit    => 'unbound.service',
      content => template("${module_name}/unbound.systemd.dropin.erb"),
      before  => Class['::unbound'],
    }
    $monit_ensure = 'absent'
  } else {
    $monit_ensure = 'present'
  }

  twitch_monit::check { 'unbound':
    ensure         => $monit_ensure,
    pidfile        => '/var/run/unbound.pid',
    check_interval => 5,
    start_program  => '/usr/sbin/service unbound start',
    stop_program   => '/usr/sbin/service unbound stop',
    custom_checks  => [
      'if failed host 127.0.0.1 port 53 type tcp protocol dns then restart',
      'if failed host 127.0.0.1 port 53 type udp protocol dns then restart',
      'if 5 restarts within 5 cycles then timeout',
    ],
    cwd            => '/',
    uid            => 'root',
    require        => Class['unbound'],
  }

  if $role {
    include "twitch_unbound::${role}"
  }

  # XXX: temporary migration step for detecting large change in num-threads
  # the unbound process needs to 'restart' in order to pickup systemd
  # LimitNOFILE overrides so each thread can have an appropraite amount of FDs
  if ($facts['unbound_num_threads'] != undef) and ($num_threads > $facts['unbound_num_threads']) {
    Service <| title == 'unbound' |> {
      restart => undef
    }

    notify { "Unbound: detected num-thread increase (${facts['unbound_num_threads']}->${num_threads}), overriding restart command":
      require => Service['unbound'],
    }
  }
}
