#!/usr/bin/python

import os, syslog
import sys
import re
import socket
import getopt
import os.path
import json
import base64
import pwd
import grp

"""Transactionally update variety of file contents and meta-data

This program is passed a file which which contains a JSON structure that
has some number of file definitions, each with contents, owner, group
and permission mode.

Usage:
$ transactional-file-update.py -i path/to/input_file.json [-v] [-d]

A simple example:
$ cat > ex1.json
{   "name" : "some-file-transaction",
    "files" : {
        "/tmp/some-key-file": {
            "value" : "adfadfads=",
            "owner" : "root",
            "group" : "root",
            "mode" : 400
        }
        "/tmp/another-key-file": {
            "value" : "ADFADFADS=",
            "owner" : "root",
            "group" : "root",
            "mode" : 400
        }
    }
}
^D
$ python transactional-file-update.py -i ex1.json

This will cause the program to first write the base64 decoded bytes in
"value" to /tmp/some-key-file.tmp, changing its owner/group to root/root,
and its mode to 400.  It does the same with /tmp/another-key-file.tmp.

If all of these operations are successful, then the program will, for
each file, atomically rename the .tmp file to the target file.

This virtually guarantees that all of the files will be updated, or none
of them will be.

A feature of this program is to report all of the errors in the case of a
failed run.  That is, it doesn't abort at the first error.

Note that this program must be run as root in order to carry out file owner
and group changes.

Required:
  -i <path to file> : Path to file containing JSON document with all of the
                      configs.

Options:
  -v : verbose  Output additional information to STDERR.
  -d : dry-run  Output to STDOUT file rename operations without doing the
                actual rename.  This is meant to allow the caller to inspect
                the .tmp files that would have been put into place.


JSON Format:
Top-level attributes:
  name(string) - name of this set of transactional file operations, used in
               reporting alerts and exceptions.
  files(dictionary) - The key is a full filesystem path and the value is a
                      File Entry

File Entry attributes:
  value(string) - base64 encoded contents of the file
  owner(string) - the program sets file owner to this
  group(string) - the program sets the file group to this
  mode(string) - the program sets the file mode to the octal cast of this.
                 For example, this might contain the string "0644"


Status Reporting

By default, this program reports diagnostics via STDOUT, STDERR and exit
code in the normal UNIX fashion.

It can also be configured to report its results via NSCA, via optional
top-level configs.

Example config:
{   "name" : "some-file-transaction",
    "files" : {
        ...
    },
    "send-nsca" : {
        "nsca-path" : "/usr/sbin/send_nsca",
        "nagios-service-name" : "Important Update"
    }
}

"""

verbose_set = False
config = False

sys_log_tag = 'transactional-file-update'
sys_log_facility = syslog.LOG_LOCAL3
sys_log_options = 0

try:
    syslog.openlog(sys_log_tag, sys_log_options,sys_log_facility)
except:
    print('failed to open syslog')
    sys.exit(2)

if not os.geteuid() == 0:
    print('program must be run as root')
    sys.exit(2)

def main(argv):
    global config
    global verbose_set
    key_file = None
    dry_run = False
    try:
        opts, args = getopt.getopt(argv, 'i:dv', ['input-file', 'dry-run', 'verbose'])
        verbose_set = False
        for o, a in opts:
            if o == '-i':
                key_file = a
            if o == '-d':
                dry_run = True
            if o == '-v':
                verbose_set = True
    except Exception as e:
        fail(str(e) + usage())
    if not key_file:
        fail(usage())
    if not os.path.exists(key_file):
        fail('key_file ' + key_file + ' does not exist')
    if not os.path.isfile(key_file):
        fail('key_file ' + key_file + ' is not a file')
    with open(key_file) as json_file:
        config = json.load(json_file)
    if not 'name' in config:
        fail('passed config does not have required "name" attribute')
    if not 'files' in config:
        fail('passed config does not have required "files" attribute')
    errors = ''
    for file_path, file_info in config['files'].iteritems():
        for required_attribute in ['value','owner','group','mode']:
            if required_attribute not in file_info:
                errors = errors + 'for file ' + file_path + ' required attribute "' + required_attribute + '" not found\n'
        tmp_file_path = file_path + '.tmp'
        verbose('file_path=' + file_path + ' tmp_file_path=' + tmp_file_path + '\n')
        if 'value' in file_info:
            try:
                if 'owner' in file_info:
                    uid = pwd.getpwnam(file_info['owner']).pw_uid
                if 'group' in file_info:
                    gid = grp.getgrnam(file_info['group']).gr_gid
                key_value = base64.b64decode(file_info['value'])
                f = open(tmp_file_path, 'wb')
                f.write(key_value)
                f.close()
                os.chown(tmp_file_path, uid, gid)
                if 'mode' in file_info:
                    mode = int(file_info['mode'], 8)
                    os.chmod(tmp_file_path, mode)
            except Exception as e:
                errors = errors + 'error handling file_path=' + file_path + ': ' + str(e)
                break
    if len(errors) != 0:
        fail(errors)

    for file_path, file_info in config['files'].iteritems():
        tmp_file_path = file_path + '.tmp'
        if dry_run:
            print('dry_run: ' + tmp_file_path + ' -> ' + file_path)
            syslog.syslog('dry_run: ' + tmp_file_path + ' -> ' + file_path)
        else:
            os.rename(tmp_file_path, file_path)
    if dry_run:
        print('dry_run: files NOT updated, .tmp files left in place for examination')
        syslog.syslog('dry_run: files NOT updated, .tmp files left in place for examination')

def usage():
    return("""USAGE: validate-tls-session-key.py -i /path/to/keyfile.json [-d]
  -i <input file path> [required] JSON file that contains necessary config
  -d [optional] If set, validates input by writing and leaving temp files for
                inspection and confirmation.
""")


def verbose(text):
    if verbose_set:
        sys.stderr.write('verbose: ' + text)

def fail(text):
    alert(text)
    sys.exit(2)

def alert(text):
    if 'send-nsca' in config:
        text = re.sub('\n', ' - ', text)
        hostname = socket.getfqdn()
        metric_string = "%s\t%s\t%s\t%s\n" % (hostname, config['send-nsca']['nagios-service-name'], 0, text)
        send_nsca = config['send-nsca']['nsca-path']
        nsca_cmd = "echo '%s' | %s -H %s" % (metric_string, send_nsca, 'video-nagios')
        print(nsca_cmd)
        syslog.syslog(nsca_cmd)

    syslog.syslog("ALERT: (not yet implemented) " + text)

if __name__ == "__main__":
    main(sys.argv[1:])
