class twitch_zookeeper (
  $client_port        = $twitch_zookeeper::params::client_port,
  $data_dir           = $twitch_zookeeper::params::data_dir,
  $election_port      = $twitch_zookeeper::params::election_port,
  $ensemble_members   = $twitch_zookeeper::params::ensemble_members,
  $init_limit         = $twitch_zookeeper::params::init_limit,
  $leader_port        = $twitch_zookeeper::params::leader_port,
  $max_client_cnxns   = $twitch_zookeeper::params::max_client_cnxns,
  $sync_limit         = $twitch_zookeeper::params::sync_limit,
  $tick_time          = $twitch_zookeeper::params::tick_time,
  $use_exhibitor      = $twitch_zookeeper::params::use_exhibitor,
  $zk_stats_interval  = $twitch_zookeeper::params::zk_stats_interval,
) inherits twitch_zookeeper::params {
  include zookeeper

  if $use_exhibitor {
    file { '/etc/zookeeper/conf/zoo.cfg':
      ensure  => 'file',
      group   => 'zookeeper',
      owner   => 'zookeeper',
      require => Package['zookeeper'],
    }
    file { '/etc/zookeeper/conf/java.env':
      ensure  => 'file',
      group   => 'zookeeper',
      owner   => 'zookeeper',
      require => Package['zookeeper'],
    }

    package { 'exhibitor':
      ensure  => '1.7.1-jtv-2',
      require => Package['zookeeper'],
    }
    ::daemontools::supervise { 'exhibitor':
      daemon     => "java -jar /opt/exhibitor/exhibitor-1.7.1.jar -c s3 --s3region=us-west-2 --s3config $::exhibitor_s3_config_file --hostname ${::ipaddress}",
      daemon_dir => '/var/lib/service',
      down       => false,
      require    => Package['exhibitor'],
      subscribe  => Package['exhibitor'],
      syslog     => 'local3',
      user       => 'zookeeper',
      wd         => '/tmp',
    }
  } else {
    $my_id = $ensemble_members[$fqdn]

    file { "${data_dir}/myid":
      content => $my_id,
      notify  => Service['zookeeper-server'],
      require => Package['zookeeper'],
    }
    file { '/etc/zookeeper/conf/zoo.cfg':
      ensure  => 'file',
      content => template("${module_name}/zoo.cfg.erb"),
      group   => 'zookeeper',
      notify  => Service['zookeeper-server'],
      owner   => 'zookeeper',
      require => Package['zookeeper'],
    }

  }

  package { 'zk-stats':
    ensure  => 'installed',
    require => Package['zookeeper'],
  }
  ::daemontools::supervise { 'zk-stats':
    ensure     => 'absent',
    daemon     => "zk-stats -cluster=$::statsdcluster -server=$::statsdserver -interval=${zk_stats_interval} -host=localhost:${client_port} >> /var/log/zookeeper/zk-stats.log 2>&1",
    daemon_dir => '/var/lib/service',
    down       => false,
    require    => Package['zk-stats'],
    subscribe  => Package['zk-stats'],
    syslog     => 'local3',
    user       => 'zookeeper',
    wd         => '/tmp',
  }

  package { 'zk-alerts':
    ensure  => 'installed',
    require => Package['zookeeper'],
  }
  ::daemontools::supervise { 'zk-alerts':
    daemon     => "zk-alerts -interval=$::alertinterval -shouldsend=$::sendalert  >> /var/log/zookeeper/zk-alerts.log 2>&1",
    daemon_dir => '/var/lib/service',
    down       => false,
    require    => Package['zk-alerts'],
    subscribe  => Package['zk-alerts'],
    syslog     => 'local3',
    user       => 'zookeeper',
    wd         => '/tmp',
  }

  logrotate::rule { 'zk-stats':
    path         => '/var/log/zookeeper/*.log /var/log/zookeeper/*.out',
    rotate_every => 'day',
    compress     => true,
    rotate       => 7,
    missingok    => true,
    copytruncate => true,
  }

  consul::service { 'zookeeper':
    check_interval => '30s',
    check_script   => 'curl http://localhost:8000/zk/state',
    consul_tags    => [$::twitch_environment, "healthcheck=http://${::fqdn}:8000/zk/state", "exhibitor=${::fqdn}:8080"],
    port           => '2181',
  }
}
