# this class handles config files for varnish
class varnish::config inherits varnish {

  # OMG AWESOME: anything that is undefined
  # will be deleted from this hash we could have done it in the template
  # puppetlabs-stdlib > 4.1.0 commit:  e49d356fbf
  # this gets put in varnish_default file
  $daemon_opts = delete_undef_values({
    '-a' => $listen,
    '-T' => $telnet_listen,
    '-f' => "${varnish_dir}/${real_config_file}",
    '-n' => $working_dir,
    '-S' => $secret_file,
    '-l' => $shared_memory,
    '-P' => $varnish_pid,
  })

  # template variables
  $daemon_extra_params = $params

  if ( $varnish_enable == true ) {
    $ensure_dir  = 'directory'
    $ensure_file = 'file'
    $ensure_else = 'present'
  } else {
    $ensure_dir  = 'absent'
    $ensure_file = 'absent'
    $ensure_else = 'absent'
  }

  # we create these directories ourselves because we make sure
  # config is placed on the system before package gets installed
  # so it starts once with all the right configuration
  file { [$varnish_dir, $vmod_base_path, $vmod_path]:
    ensure => $ensure_dir,
    owner  => 'root',
    group  => 'root',
    mode   => '0755',
  } ->

  exec { $secret_file:
    command => "dd if=/dev/urandom count=1 bs=512 status=none > '${secret_file}'; chmod 600 '${secret_file}'",
    creates => $secret_file,
    user    => 'root',
    group   => 'root',
  } ->

  user { $varnishlog_uid:
    ensure => 'present',
    system => true,
    shell  => '/bin/false'
  } ->

  file { $varnish_default:
    ensure  => $ensure_file,
    owner   => 'root',
    group   => 'root',
    content => template("${module_name}/${release}/default_varnish.erb"),
    notify  => $varnish_notify ? {
      true  => Service[$varnish_service],
      false => undef
    }
  } ->

  file { $varnishncsa_default:
    ensure  => $ensure_file,
    owner   => 'root',
    group   => 'root',
    content => template("${module_name}/${release}/default_varnishncsa.erb"),
    notify  => $varnishncsa_notify ? {
      true  => Service[$varnishncsa_service],
      false => undef
    }
  } ->

  file { $varnishlog_default:
    ensure  => $ensure_file,
    owner   => 'root',
    group   => 'root',
    content => template("${module_name}/${release}/default_varnishlog.erb"),
    notify  => $varnishlog_notify ? {
      true  => Service[$varnishlog_service],
      false => undef
    }
  } ->

  # we must have this here, because some people wanted to edit it for no reason
  # this is the default init script shipped with 3.0.4~1 varnish for ubuntu
  file { '/etc/init.d/varnishncsa':
    ensure => $ensure_file,
    source => "puppet:///modules/${module_name}/${release}/init_varnishncsa",
    owner  => 'root',
    group  => 'root',
    mode   => '0755',
  }

  if str2bool($::systemd) {
    file { '/etc/varnish/varnishncsa.params':
      ensure  => $ensure_file,
      owner   => 'root',
      group   => 'root',
      mode    => '0644',
      content => template("${module_name}/${release}/varnishncsa.params.erb"),
      require => File['/etc/init.d/varnishncsa'],
    } ->

    twitch_systemd::unit_file { 'varnishncsa.service':
      ensure  => $ensure_else,
      content => template("${module_name}/${release}/varnishncsa.service.erb"),
      notify  => Service[$varnishncsa_service],
    } ->

    file { '/etc/varnish/varnishlog.params':
      ensure  => $ensure_file,
      owner   => 'root',
      group   => 'root',
      mode    => '0644',
      content => template("${module_name}/${release}/varnishlog.params.erb"),
    } ->

    twitch_systemd::unit_file { 'varnishlog.service':
      ensure  => $ensure_else,
      content => template("${module_name}/${release}/varnishlog.service.erb"),
    } ->

    file { '/etc/varnish/varnish.params':
      ensure  => $ensure_file,
      owner   => 'root',
      group   => 'root',
      mode    => '0644',
      content => template("${module_name}/${release}/varnish.params.erb"),
    } ->

    twitch_systemd::unit_file { 'varnish.service':
      ensure  => $ensure_else,
      content => template("${module_name}/${release}/varnish.service.erb"),
    }
  }
  file { '/usr/bin/varnishstat_wrapper.py':
    ensure => $ensure_file,
    source => "puppet:///modules/${module_name}/${release}/varnishstat_wrapper.py",
    owner  => 'root',
    group  => 'root',
    mode   => '0755',
  }

  # this will install the defined vcl
  # if config_template, use that
  # elseif use config_file
  if ($config_template) {
    varnish::vcl { $real_config_file:
      ensure   => $ensure_else,
      template => $config_template
    }
  } elsif ($config_file) {
    varnish::vcl { $real_config_file:
      ensure => $ensure_else,
      source => $config_file
    }
  }
}
