# Class varnish
#
#  Installs Varnish, and creates default directories and services
#
#  Usage:
#  class { '::varnish':
#     version         => '3:6.0.1-1+%{::lsbdistcodename}+twitch',
#     release         => '6.0',
#     telnet_listen   => '127.0.0.1:2000',
#     backend_storage => 'malloc,4G',
#     worker_threads  => '100,2000,300',
#     config_file     => 'countess.vcl',
#  }
#
class varnish(
  $package            = $varnish::params::package,
  $release            = $varnish::params::varnish_release,
  $version            = $varnish::params::varnish_version,

  $nfiles             = $varnish::params::nfiles,
  $memlock            = $varnish::params::memlock,

  $listen             = $varnish::params::listen,
  $telnet_listen      = $varnish::params::telnet_listen,
  $config_file        = $varnish::params::config_file,
  $config_template    = $varnish::params::config_template,
  $working_dir        = $varnish::params::working_dir,
  $worker_threads     = $varnish::params::worker_threads,
  $secret_file        = $varnish::params::secret_file,
  $backend_storage    = $varnish::params::backend_storage,
  $shared_memory      = $varnish::params::shared_memory,
  $params             = $varnish::params::params,

  $varnish_enable     = $varnish::params::varnish_enable,
  $varnish_notify     = $varnish::params::varnish_notify,
  $varnish_dir        = $varnish::params::varnish_dir,
  $varnish_pid        = $varnish::params::varnish_pid,

  $varnishlog_enable  = $varnish::params::varnishlog_enable,
  $varnishlog_notify  = $varnish::params::varnishlog_notify,
  $varnishlog_gid     = $varnish::params::varnishlog_gid,

  $varnishncsa_enable = $varnish::params::varnishncsa_enable,
  $varnishncsa_notify = $varnish::params::varnishncsa_notify,

  $log_format         = $varnish::params::log_format,
  $vsl_query          = $varnish::params::vsl_query,
  $varnish_pop        = $varnish::params::varnish_pop,

  $segment_ttls       = $varnish::params::segment_ttls,
  $upstream_override  = $varnish::params::upstream_override,

  $playlist_longpoll  = $varnish::params::playlist_longpoll,
  $varnish_nrdp_args  = $varnish::params::varnish_nrdp_args,
  $numa_interleave    = $varnish::params::numa_interleave,
  $upstream_reporting = $varnish::params::upstream_reporting,

  $varnishd_syslog_enable = $varnish::params::varnishd_syslog_enable,
) inherits varnish::params {

  # mutually exclusive check
  if ($config_template) and ($config_file) {
    fail('Cannot specify both config_file and config_template')
  }

  # if config_template passed in, use it
  if ($config_template) {
    validate_re($config_template, '\.erb$')
    # we have a template, strip out the erb portion, get name of file
    $real_config_file = regsubst(basename($config_template), '\.erb$', '')
  # otherwise use the config_file
  } elsif ($config_file) {
    # validate that config_file passed in is at least one non-space character
    validate_re($config_file, '^(?!\s*$).+')
    # we have a config_file so lets get basename
    $real_config_file = basename($config_file)
  } else {
    fail('Varnish needs either $config_file or $config_template passed in')
  }

  # lets make sure we stick to conventions, configs end with vcl
  # ensure it doesn't have a slash in it
  validate_re($real_config_file, '^[^\.\/]+\.vcl$')

  # boolean validation
  validate_bool($varnish_enable,
    $varnishlog_enable,
    $varnishncsa_enable,
    $varnish_notify,
    $varnishlog_notify,
    $varnishncsa_notify,
    $playlist_longpoll,
    $upstream_reporting)

  # validate user provided hash
  validate_hash($params)

  # lets use some validations on files
  validate_absolute_path($secret_file, $varnish_dir)
  if ($working_dir) {
    validate_absolute_path($working_dir)
  }

  # install default vmods declared in params.pp
  include varnish::vmods

  anchor { 'varnish::begin': } ->
  class { '::varnish::config': } ->
  class { '::varnish::logging': } ->
  class { '::varnish::install': } ->
  class { '::varnish::config_dirs': } ->
  class { '::varnish::services': } ->
  anchor { 'varnish::end': }

  # XXX: xenial is missing this .orig file, which is what the varnish_wrapper.py uses currently.
  # https://git-aws.internal.justin.tv/systems/puppet/blob/66fa2e79ba06a1b2f8b7cf87691393d5e785705b/modules/varnish/files/4.1/varnishstat_wrapper.py#L6
  exec { 'copy original varnishstat binary':
    command   => 'cp /usr/bin/varnishstat /usr/bin/varnishstat.orig',
    creates   => '/usr/bin/varnishstat.orig',
    subscribe => Package[$package]
  }

  # Nagios NRDP check watching varnish stats
  @twitch_nrdp::check { 'check_varnish_stats':
    service_name => 'check_varnish_stats',
    command      => "/usr/local/monitor_scripts/check_varnish_stats.py ${varnish_nrdp_args}",
  }

}
