# Installs and manages config of video's certzero scanner
class video_certzero (
  $ensure    = $video_certzero::params::ensure,
  $package   = $video_certzero::params::package,
  $frequency = $video_certzero::params::frequency,
  $splay     = $video_certzero::params::splay,
  $milliner  = $video_certzero::params::milliner,
) inherits video_certzero::params {

  include video_cloudwatch::emf

  if $ensure == 'absent' {
    $rsrc_ensure = 'absent'
    $svc_ensure  = 'stopped'
    $svc_enable  = 'false'
    $ve_req      = Notify['absent']
  } else {
    include twitch_pip::pip3

    $rsrc_ensure = 'present'
    $svc_ensure  = 'running'
    $svc_enable  = 'true'
    $ve_req      = [ Exec['cleanup'], Class['twitch_pip::pip3'] ]
  }

  $venv = '/opt/twitch/certzero/venv'

  if $rsrc_ensure == 'present' {
    exec { 'cleanup':
        provider => 'shell',
        unless   => "[ -x ${venv} ] && [ -x ${venv}/bin/pip3 ] && ${venv}/bin/pip3 --help && [ -x ${venv}/bin/easy_install ] && ${venv}/bin/easy_install --help",
        command  => "rm -rf ${venv}",
    }
  } else {
    notify { 'absent': }
  }

  if $::lsbdistcodename == 'bionic' {
    python::virtualenv { $venv:
      ensure     => $rsrc_ensure,
      version    => '3.6',
      index      => 'https://pypi.python.org/simple/',
      systempkgs => true,
      distribute => false,
      virtualenv => 'virtualenv --no-download',
      require    => $ve_req,
    }
  } else {
    python::virtualenv { $venv:
      ensure     => $rsrc_ensure,
      version    => '3',
      systempkgs => true,
      distribute => false,
      virtualenv => 'virtualenv --no-download',
      require    => $ve_req,
    }
  }

  if $rsrc_ensure == 'present' {
    # support switching between staging and prod pkgs
    $rm_pkg = $package ? {
      'video-certzero' => 'video-certzero-staging',
      default          => 'video-certzero'
    }
    python::pip { $rm_pkg:
      ensure     => absent,
      virtualenv => $venv,
      require    => Python::Virtualenv[$venv],
    }

    python::pip { $package:
      ensure     => $ensure,
      virtualenv => $venv,
      index      => 'https://pypi.internal.justin.tv/simple',
      require    => [ Python::Virtualenv[$venv], Python::Pip[$rm_pkg] ],
    }

    $svc_req = Python::Pip[$package]

    if $milliner {
      require milliner
      $svc_after = 'video-milliner.service'
    }
  }

  systemd::service { 'video-certzero':
    ensure            => $rsrc_ensure,
    description       => 'certzero periodic certificate scan',
    documentation     => 'https://git.xarth.tv/video-coreservices/certzero',
    type              => 'oneshot',
    after             => $svc_after,
    environment_files => [
        '-/etc/profile.d/proxy.sh',
        '-/etc/milliner-client-env.sh',
    ],
    env_vars          => [
        "AWS_DEFAULT_REGION=${::default_region}",
        'CRYPTOGRAPHY_ALLOW_OPENSSL_102=1',
        'AWS_STS_REGIONAL_ENDPOINTS=regional',
        'AWS_CSM_ENABLED=true',
        'AWS_CSM_CLIENT_ID=CertZero',
    ],
    execstart         => "${venv}/bin/filesystem_scan --pop ${::pop}",
    require           => $svc_req,
    notify            => Class['systemd'],
  }

  systemd::timer { 'video-certzero':
    ensure               => $rsrc_ensure,
    on_boot_sec          => $frequency,
    on_unit_active_sec   => $frequency,
    randomized_delay_sec => $splay,
    description          => 'certzero periodic certificate scan',
    documentation        => 'https://git.xarth.tv/video-coreservices/certzero',
    wants                => [ 'video-certzero.service' ],
    require              => Systemd::Service['video-certzero'],
  }

  service { 'video-certzero.timer':
    ensure  => $svc_ensure,
    enable  => $svc_enable,
    require => [ Systemd::Timer['video-certzero'], Class['systemd'] ],
  }

}
