# video_codedeploy ensures that the latest versions of the video-codedeploy-agent and
# video-codedeploy-harness packages are installed and running. It also ensures that
# the latest video-winc and video-codedeploy-prune packages are installed.

# Defaults to ensuring the `latest` version.
class video_codedeploy (
  $version = $video_codedeploy::params::version,
  $ensure  = $video_codedeploy::params::ensure,
) inherits video_codedeploy::params {

  require twitch_amazon_ssm_agent
  require milliner

  # Set the package ensure parameter to absent or the version provided.
  case $ensure {
    present: {
      $ensure_pkg = $version
      $ensure_svc = running
      $enable_svc = true
    }
    default: {
      $ensure_pkg = absent
      $ensure_svc = stopped
      $enable_svc = false
    }
  }

  # add packages
  package { 'video-codedeploy-agent':
    ensure      => $ensure_pkg,
    configfiles => 'replace',
  }

  package { 'video-codedeploy-harness':
    ensure      => $ensure_pkg,
    configfiles => 'replace',
  }

  # add WINC utility (not a service)
  package { 'video-winc':
    ensure      => $ensure_pkg,
    configfiles => 'replace',
  }

  # Using Unattended-Upgrade::Package-Whitelist causes all other packages to
  # not update. Use a cronjob instead. Delete this once all hosts run puppet
  # agent on a schedule.
  cron::job { 'video-winc':
    ensure      => 'present',
    command     => 'apt-get install -y video-winc >/dev/null 2>&1',
    user        => 'root',
    environment => [
      'DEBIAN_FRONTEND=noninteractive',
      'PATH=/bin:/usr/bin:/sbin:/usr/sbin:/usr/local/bin:/usr/local/sbin',
    ],
    minute      => fqdn_rand(59, $title),
    hour        => fqdn_rand(23, $title),
  }

  # add video-codedeploy-prune utility (not a service)
  package { 'video-codedeploy-prune':
    ensure      => $ensure_pkg,
    configfiles => 'replace',
  }

  # add services
  service { 'video-codedeploy-agent':
    ensure  => $ensure_svc,
    enable  => $enable_svc,
    require => Package['video-codedeploy-agent'],
  }

  service { 'video-codedeploy-harness':
    ensure  => $ensure_svc,
    enable  => $enable_svc,
    require => Package['video-codedeploy-harness'],
  }

  # populate config for vid-cs CWA to slurp the agent logs
  video_cloudwatch::logfile { 'codedeploy-agent':
    file_path        => '/var/log/aws/codedeploy-agent/codedeploy-agent.log',
    log_group_name   => '/vidtools/codedeploy-agent',
    timestamp_format => '%Y-%m-%d %H:%M:%S',
    timezone         => 'UTC',
  }

  package { 'check-systemd-service':
    ensure => latest,
  }

  twitch_servicecheck::passive { 'check_codedeploy_status':
    command        => '/usr/local/bin/check-systemd-service -s video-codedeploy-agent',
    interval       => 5, # minutes between checks
    retry          => 3, # times to fail before reporting failure
    retry_interval => 2, # minutes between checks once reporting failure
  }

  twitch_servicecheck::passive { 'check_codedeploy_harness_status':
    command        => '/usr/local/bin/check-systemd-service -s video-codedeploy-harness',
    interval       => 5, # minutes between checks
    retry          => 3, # times to fail before reporting failure
    retry_interval => 2, # minutes between checks once reporting failure
  }

}
