# == Class: video_origin
#
# video_origin runs the code.justin.tv/video/origin/cmd/origin binary, which is
# an HLS origin server: it receives segments and playlist from the
# TwitchTranscoder and makes them available over HTTP GET requests.

class video_origin (
  $origin_env               = $video_origin::params::origin_env,
  $canary                   = $video_origin::params::canary,
  $include_event_transcoder = $video_origin::params::include_event_transcoder,
  $include_transcoder       = $video_origin::params::include_transcoder,
  $include_transmuxer       = $video_origin::params::include_transmuxer,
  $origin_dir               = '/dev/shm/origin',
  $max_bytes                = $video_origin::params::max_bytes,
) inherits video_origin::params {
  # declare dependent services close to the service definition
  include twitch::haproxy::backend
  include twitch::haproxy::listeners::backend_s3_http_proxy
  include twitch::haproxy::listeners::backend_rabbit_cluster_hls_production
  include twitch::haproxy::listeners::backend_video_api_http
  # temporary fix for conflicting module inclusion
  if $include_event_transcoder {
    include video::workers::hls_event_transcoder
  } else {
    if $include_transcoder {
      include video::workers::hls_transcoder
    }
    if $include_transmuxer {
      include video::workers::hls_transmuxer
    }
  }

  # Values
  $env = $::twitch_environment
  $origin_dc = $::pop
  $role = $::twitch_role
  $base_path = '/opt/twitch/video-origin/current'
  $cfg_dir = '/etc/origin'
  $cfg_file = "${cfg_dir}/origin.json"
  $user = 'video-origin'
  # Needed for twitch_svclogs unless default_region fact can be set to these mappings
  $bound_region = $origin_dc ? {
    'lhr05' => 'eu-west-2',
    default => $::default_region,
  }
  $origin_port = '9090'
  $origindb_port = '9091'
  $session_dir = "${origin_dir}/sessions"
  $blob_dir = "${origin_dir}/blobs"

  # Validate variable values
  validate_string($env, $origin_dc, $role)
  validate_re($env, '^(production|staging|dev|canary-c|canary-d)$', 'env invalid, must be {production,staging,dev,canary-c,canary-d}')
  validate_re($role, '^transcode(|-x2|-qs|-vp9|-ev)$', 'role invalid, must be {transcode,transcode-x2,transcode-qs,transcode-vp9,transcode-ev}')

  $env_flags = join([
    "-role=${::twitch_role}",
    "-environment=${::twitch_environment}",
    "-pop=${::pop}",
  ], ' ')

  # Set up initial config that is passed into origin
  file { $cfg_dir:
    ensure  => directory,
    path    => $cfg_dir,
  }

  $cfg = delete_undef_values({
    'ArtifactDir'  => $base_path,
    'BlobDir'      => $blob_dir,
    'BoundRegion'  => $bound_region,
    'Environment'  => $env,
    'MaxBytes'     => $max_bytes,
    'OriginDC'     => $origin_dc,
    'OriginPort'   => $origin_port,
    'OriginDBPort' => $origindb_port,
    'Role'         => $role,
    'SessionDir'   => $session_dir,
  })

  file { $cfg_file:
    ensure  => file,
    path    => $cfg_file,
    content => pretty_json($cfg),
    require => File[$cfg_dir],
  }

  twitch::service { 'video-origin':
    env             => $origin_env,
    canary          => $canary,
    daemon_cmd      => join([
      'bin/origin',
      "--config ${cfg_file}",
      "-port=${origin_port}",
      "-origindb-addr=localhost:${origindb_port}",
      "-session-dir=${session_dir}",
      $env_flags,
    ], ' '),
    service_repo    => 'video/origin',
    predaemon       => [template("${module_name}/predaemon_origin.erb")],
    syslog          => 'local5',
    user            => $user,
    courier_install => false,
  }

  $base_dir = hiera('twitch_basedir')
  daemontools::supervise { 'video-origindb':
    ensure      => 'present',
    daemon_dir  => '/var/lib/service',
    daemon      => join([
      'bin/origindb',
      "--config ${cfg_file}",
      "-port=${origindb_port}",
      "-wd=${blob_dir}",
      "-max-bytes=${max_bytes}",
      $env_flags,
      '2>&1',
    ], ' '),
    wd          => "${base_dir}/video-origin/current",
    syslog      => 'local5',
    user        => $user,
    env         => $origin_env,
    concurrency => 0,
    predaemon   => [template("${module_name}/predaemon_origindb.erb")],
    require     => Twitch::Service['video-origin'],
  }

  class { 'twitch_crashmonitor':
    ensure => 'absent',
  }

}
