# Configures a video-tier1 host.

class video_tier1 (
  $svc_name              = 'hlspr',
  $consul_env            = $twitch_environment,
  $consul_tags           = $video_tier1::params::consul_tags,
  $nginx_reload          = $video_tier1::params::nginx_reload,
  $use_cloudwatch        = $video_tier1::params::use_cloudwatch,
  $varnish_release       = $video_tier1::params::varnish_release,
  $varnish_pop           = $video_tier1::params::varnish_pop,
  $varnish_daemon_params = $video_tier1::params::varnish_daemon_params,
  $use_varnish           = $video_tier1::params::use_varnish,
  $frontier_transition   = $video_tier1::params::frontier_transition,
) inherits video_tier1::params {
  include twitch_nrdp
  include twitch_vars
  include twitch::crossdomain
  include twitch::packages
  include twitch_python::pip::statsd
  include video_tier1::nginx_graphite
  # varnish specific include
  include libvmod_timers2

  if !$frontier_transition {
    # manage duplicate consul service if frontier_transition hasn't begun 
    consul::service { 'video-hls-replication':
      port        => 80,
      consul_tags => concat($consul_tags, "fqdn=${::fqdn}", 'role-t1', $consul_env)
    }
  }

  # variables used only to populate templates
  $nginx_conf_dir          = $twitch_nginx::params::nginx_conf_dir
  $nameservers             = hiera('twitch_nameservers')
  $tier1_nginx_port        = 9080

  # extra, static-ish manifest variables
  $base_dir                = hiera('twitch_basedir', '/opt/twitch')
  $video_tier1_dir         = "${base_dir}/video_tier1"
  $video_tier1_conf_dir    = "${video_tier1_dir}/etc"
  $upstreams_template      = "${video_tier1_conf_dir}/upstreams.ctmpl"
  $tier1_nginx_template    = "${module_name}/nginx_${svc_name}_video_tier1.conf.erb"
  $allow_ips               = concat($::twitch_vars::twitch_networks, '127.0.0.0/8')

  validate_array($consul_tags)
  validate_bool($use_cloudwatch)
  validate_bool($frontier_transition)

  # Used in hlspr nginx template
  if versioncmp($::kernelmajversion, '3.9') >= 0 {
    $reuseport_supported = true
  } else {
    $reuseport_supported = false
  }
  # cloudwatch logging
  if ($use_cloudwatch) {
    $_use_cloudwatch_ensure = 'present'
  } else {
    $_use_cloudwatch_ensure = 'absent'
  }

  twitch_svclogs::logfile { 'nginx-error':
    ensure          => $_use_cloudwatch_ensure,
    datetime_format => '%d/%b/%Y:%H:%M:%S %z',
    log_group_name  => "/tier1/${::pop}/nginx-error",
    log_dir         => '/var/log/nginx',
    filename        => 'video_tier1_pr_error.log',
  }

  file { [$video_tier1_dir, $video_tier1_conf_dir]:
    ensure => 'directory',
    owner  => 'jtv',
  }

  file { $upstreams_template:
    owner  => 'jtv',
    group  => 'jtv',
    source => 'puppet:///modules/video_tier1/nginx/upstreams.ctmpl',
  }

  ::twitch_nginx::vhost{ $svc_name:
    content => template($tier1_nginx_template),
  }

  file { '/home/jtv/cronscripts/tier1_error_checking.py':
    ensure  => present,
    require => File['/home/jtv/cronscripts/'],
    owner   => 'jtv',
    group   => 'jtv',
    mode    => '0755',
    source  => 'puppet:///modules/video_tier1/tier1_error_checking.py',
  }

  cron::job { 'tier1_error_checking':
    command => '/home/jtv/cronscripts/tier1_error_checking.py 2>&1 | logger -t tier1_errchk_cron -p local3.info',
    user    => 'nagios',
    minute  => '*/1',
  }

  # crashmonitor bites
  realize(Twitch_nrdp::Check['check_sandstorm_agent','check_sandstorm_agent_ops'])
  class { 'twitch_crashmonitor': }

  # varnish config below
  $statsd_sample_rate = 0.05
  # which port to configure varnish with
  $tier1_varnish_port = 80
  $vcl_files = {
    'common.vcl' => {
      template => 'hls/common.vcl.erb'
    },
    'base.vcl' => {
      'template' => 'hls/base.vcl.erb',
    },
  }
  $vcl_properties = {}

  create_resources(varnish::vcl, $vcl_files, $vcl_properties)

  class { '::varnish':
    release            => $varnish_release,
    listen             => ":${tier1_varnish_port}",
    working_dir        => '/dev/shm/varnish',
    log_format         => '%h %l %u %t "%r" %s %b "%{Referer}i" "%{User-agent}i" %D %{Varnish:time_firstbyte}x %{Varnish:handling}x "%{VCL_Log:gk}x" "%{VCL_Log:replication_timestamps}x"',
    config_template    => 'hls/video-tier1.vcl.erb',
    varnishncsa_enable => true,
    backend_storage    => 'malloc,50g',
    shared_memory      => '80m,10m',
    params             => $varnish_daemon_params,
  }

  # We create a crojob that runs this file hourly, so the daily rotation is a duplicate.
  # We have to set rotate->72 to keep 72 hours of logs using the cron defined elsewhere.
  logrotate::rule{ 'varnish':
    ensure          => present,
    path            => '/var/log/varnish/varnish*.log',
    missingok       => true,
    rotate          => 72,
    rotate_every    => 'day',
    sharedscripts   => true,
    compress        => true,
    delaycompress   => false,
    compresscmd     => '/bin/gzip',
    compressoptions => '-1',
    compressext     => '.gz',
    create          => true,
    create_mode     => '0644',
    create_owner    => 'varnishlog',
    postrotate      => join([
      'for service in varnishlog varnishncsa; do',
      '      if /usr/bin/pgrep -P 1 $service > /dev/null 2>&1; then',
      '        /usr/sbin/service $service reload > /dev/null 2>&1',
      '      fi',
      'done',
      'if systemctl -q is-active awslogs; then',
      '  /usr/sbin/service awslogs restart > /dev/null 2>&1',
      'fi' ],"\n"),
    require         => Class['::varnish'],
  }

  class { 'varnishshmparser':
    node_type          => 'tier1',
    statsd_sample_rate => $statsd_sample_rate,
  }
}
