# Install the warp service
class warp (
  $ensure = $warp::params::ensure,
  $env    = $warp::params::env,
) inherits warp::params {
  validate_re($ensure, [ '^present$', '^absent$' ])

  if $ensure == 'present' {
    $ensure_service = 'running'
    $ensure_directory = 'directory'
    $ensure_enabled = 'true'
  } else {
    $ensure_service = 'stopped'
    $ensure_directory = 'absent'
    $ensure_enabled = 'false'
  }

  $user = 'warp'
  $conf_dir = '/etc/warp'
  $log_dir = '/var/log/warp'

  # Create the warp user
  twitch::local_user { $user:
    group => $user,
  }

  # Setup the config directory
  file { $conf_dir:
    ensure  => $ensure_directory,
    path    => $conf_dir,
    owner   => $user,
    group   => $user,
    require => Twitch::Local_user[$user],
  }

  # Save verbose qlog traces in a folder we regularly trim.
  file { $log_dir:
    ensure => $ensure_directory,
    path   => $log_dir,
    owner  => $user,
    group  => $user,
    require => Twitch::Local_user[$user],
  }

  # Delete any longs that haven't been modified in three days
  systemd::service { 'warp-cleanup':
    ensure      => $ensure,
    description => "deletes any files in ${log_dir} older than three days",
    type        => 'oneshot',
    user        => $user,
    execstart   => "/usr/bin/find ${log_dir} -mtime +2 -type f -delete",
  }

  systemd::timer { 'warp-cleanup':
    ensure             => $ensure,
    on_boot_sec        => "15m",
    on_unit_active_sec => "1d",
    description        => "deletes any files in ${log_dir} older than a week",
    wants              => [ 'warp-cleanup.service' ],
    require            => Systemd::Service['warp-cleanup'],
  }

  service { 'warp-cleanup.timer':
    ensure  => $ensure_service,
    enable  => $ensure_enabled,
    require => Systemd::Timer['warp-cleanup'],
  }

  # Setup the TLS certs
  # *.<pop>.warp.live-video.net cert
  $key_prefix = "video/video-edge/production/${::pop}/wildcard_${::pop}_warp_live-video_net"

  $rsa_cert_contents = join([
    "{{ key \"${key_prefix}/cert\" }}",
    "{{ key \"${key_prefix}/intermediate\" }}",
  ], "\n")

  twitch_ssl_certificate::certificate { 'warp':
    ensure               => $ensure,
    cert_filesystem_path => "${conf_dir}/warp_live_video.crt",
    cert_contents        => $rsa_cert_contents,
    cert_owner           => $user,
    cert_group           => 'ssl-cert',
    cert_mode            => '0640',
    cert_restart_cmd     => 'systemctl try-reload-or-restart warp',
    key_filesystem_path  => "${conf_dir}/warp_live_video.key",
    key_contents         => "{{ key \"${key_prefix}/key\" }}",
    key_owner            => $user,
    key_group            => 'ssl-cert',
    key_mode             => '0640',
    key_restart_cmd      => 'systemctl try-reload-or-restart warp',
    require              => File[$conf_dir],
  }

  # monitoring
  twitch_servicecheck::passive { 'Warp Health':
    command        => 'check_http -H 127.0.0.1 -u /health -p 9001 -e 200,429',
    interval       => 1,
    retry          => 2,
    retry_interval => 1,
  }

  consul::service { 'warp':
    ensure         => $ensure,
    port           => 9001,
    consul_tags    => [ $env ],
    check_http     => "http://localhost:9001/health",
    check_interval => '5s',
  }

  # Generate the systemd template to execute this command.
  $warp_cmd = "/opt/twitch/warp/current/bin/warp-server --env ${env}"

  twitch_systemd::unit_file {'warp.service':
    ensure  => $ensure,
    content => template('warp/warp.service.erb'),
    require => Class['milliner'],
  }
}
