class weaver (
  $ensure                = $weaver::params::ensure,
  $primary_public_addr   = $weaver::params::primary_public_addr,
  $primary_private_addr  = $weaver::params::primary_private_addr,
  $backup_public_addr    = $weaver::params::backup_public_addr,
  $backup_private_addr   = $weaver::params::backup_private_addr,
  $env                   = $weaver::params::env,
  $basedir               = $weaver::params::basedir,
  $project               = $weaver::params::project,

  $statsd_addr           = $weaver::params::statsd_addr,
  $statsd_type           = $weaver::params::statsd_type,
  $statsd_rate           = $weaver::params::statsd_rate,

  $sandstorm_prefix      = $weaver::params::sandstorm_prefix,
  $aws_access_key_path   = $weaver::params::aws_access_key_path,

  $upstream_url          = $weaver::params::upstream_url,
  $ads_ehrmantraut       = $weaver::params::ads_ehrmantraut,
  $ads_fring             = $weaver::params::ads_fring,
  $ad_concurrency_limit  = $weaver::params::ad_concurrency_limit,

  $redis_protocol        = $weaver::params::redis_protocol,
  $redis_address         = $weaver::params::redis_address,
  $redis_cluster         = $weaver::params::redis_cluster,
  $redis_cluster_options = $weaver::params::redis_cluster_options,

  $haproxy_checks        = $weaver::params::haproxy_checks,
  $haproxy_checks_enable = $weaver::params::haproxy_checks_enable,
  $server_status         = $weaver::params::server_status,

  $beacon_url            = $weaver::params::beacon_url,

  $enable_longpoll       = $weaver::params::enable_longpoll,

  $gatekeeper_library    = $weaver::params::gatekeeper_library,
  $gatekeeper_name       = $weaver::params::gatekeeper_name,
  $gatekeeper_fqdn       = $weaver::params::gatekeeper_fqdn,

  $rubick_masters        = $weaver::params::rubick_masters,
  $rubick_replicas       = $weaver::params::rubick_replicas,
) inherits weaver::params {
  validate_re($ensure, [ '^present$', '^absent$' ])
  validate_bool($redis_cluster, $enable_longpoll, $gatekeeper_library)
  validate_re($env, ['^production$', '^staging$', '^dev$', '^canary$'])
  validate_string($redis_protocol, $redis_address, $gatekeeper_name, $gatekeeper_fqdn)

  realize(Twitch_nrdp::Check['check_sandstorm_agent','check_sandstorm_agent_ops'])

  include twitch_nrdp

  if ($server_status) {
    # Checks if consul, consul deploys, and rubick are all running
    twitch_nrdp::check { 'server-status':
      ensure   => $ensure,
      command  => '/usr/local/monitor_scripts/check_consul_service_enabled.sh "[r]ubick.conf" "Rubick"',
      interval => 1,
      retry    => 3,
    }
  }

  if ($redis_protocol == 'unix') {
    validate_absolute_path($redis_address)
  }

  $consul_tags = $env ? {
    'canary' => [ $env, 'production' ],
    default  => [ $env ],
  }

  if ($haproxy_checks_enable) {
    twitch_nrdp::check { 'weaver_haproxy_stats':
      command  => "check_haproxy_stats.py -s '/var/run/' -g 'haproxy-video_weaver-stats.1.sock' -c $haproxy_checks",
      interval => 1,
      retry    => 3,
    }
  }

  include twitch_redis

  twitch_sandstorm_agent::template { 'aws_access_key_path':
    destination     => $aws_access_key_path,
    ensure          => $ensure,
    owner           => 'root',
    group           => 'root',
    mode            => '0400',
    restart_command => '/usr/bin/svc -h /etc/service/weaver',
    contents        => join([
      "export AWS_ACCESS_KEY_ID={{ key \"${sandstorm_prefix}/${env}/all/weaver_aws_access_key_id\" }}",
      "export AWS_SECRET_ACCESS_KEY={{ key \"${sandstorm_prefix}/${env}/all/weaver_aws_secret_access_key\" }}",
      "export REDIS_PASSWORD={{ key \"video/rubick/${::twitch_environment}/redis_password\" }}",
    ], "\n")
  }

  if $ensure == 'present' {
    Twitch::Local_user['weaver'] -> Weaver::Instance['weaver', 'weaver_backup']
  } else {
    Weaver::Instance['weaver', 'weaver_backup'] -> Twitch::Local_user['weaver']
  }

  # create a local user and group, add to redis group to access socket
  # XXX: not needed if we goto TCP socket
  twitch::local_user { 'weaver':
    ensure  => $ensure,
    groups  => 'redis',
    require => Class['twitch_redis'],
  }

  if ($redis_cluster) {
    # TODO: Delete
    $redis_ports     = range('6379','6382')
    $redis_instances = prefix($redis_ports, 'weaver-')

    # TODO: Delete
    weaver::redis_cluster_instance { $redis_instances:
      ensure          => absent,
      consul_tags     => $consul_tags,
      default_options => $redis_cluster_options,
      before          => Weaver::Instance['weaver', 'weaver_backup'],
    }

    package { 'redis':
      ensure   => $ensure ? {
        'absent' => 'absent',
        default   => '3.3.3',
      },
      provider => 'gem',
    }

    # Rubick creates and manages a redis cluster.
    class { 'rubick':
      ensure   => $ensure,
      port     => 6379,
      masters  => $rubick_masters,
      replicas => $rubick_replicas,
    }
  }

  $ensure_instance = $ensure ? {
    'absent' => 'absent',
    default => $redis_cluster ? {
      true => 'absent',
      default => 'present',
    }
  }

  # This is the single instance of Redis, not managed by Rubick.
  # Ensure absent for cluster to remove some config files.
  twitch_redis::server { "weaver":
    ensure         => $ensure_instance,
    usesocket      => true,
    unixsocketperm => 777,
    bind           => "127.0.0.1", # only bind to localhost
    port           => 6379,
    manage_service => false,
    save           => [ '""' ], # no saving
    before         => [
      Twitch::Local_user['weaver'],
      Weaver::Instance['weaver'],
      Weaver::Instance['weaver_backup']
    ]
  }

  twitch_redis::server_metrics { 'weaver_6379':
    ensure => $ensure_instance,
    addr   => 'localhost:6379',
  }

  # default parameters for primary and backup instance
  Weaver::Instance {
    ensure             => $ensure,
    consul_tags        => $consul_tags,
    upstream_url       => $upstream_url,
    redis_protocol     => $redis_protocol,
    redis_address      => $redis_address,
    redis_cluster      => $redis_cluster,
    beacon_url         => $beacon_url,
    statsd_addr        => $statsd_addr,
    statsd_type        => $statsd_type,
    statsd_rate        => $statsd_rate,
    env                => $env,
    enable_longpoll    => $enable_longpoll,
    gatekeeper_library => $gatekeeper_library,
    gatekeeper_name    => $gatekeeper_name,
    gatekeeper_fqdn    => $gatekeeper_fqdn,
    ad_concurrency_limit => $ad_concurrency_limit,
  }

  weaver::instance {
    'weaver':
      public_addr  => $primary_public_addr,
      private_addr => $primary_private_addr,
      backup       => false;

    'weaver_backup':
      public_addr  => $backup_public_addr,
      private_addr => $backup_private_addr,
      backup       => true;
  }
}
