define weaver::instance(
  $ensure,
  $public_addr,
  $private_addr,
  $backup,
  $consul_tags,
  $redis_address,
  $redis_protocol,
  $redis_cluster,
  $upstream_url,
  $beacon_url,
  $statsd_addr,
  $statsd_type, # TODO remove when weaver is removed from edge
  $statsd_rate,
  $env,
  $enable_longpoll,
  $gatekeeper_library,
  $gatekeeper_name,
  $gatekeeper_fqdn,
  $ad_concurrency_limit, # For TNF2020 load test
) {
  validate_bool($backup)
  validate_string($public_addr)
  validate_string($private_addr)
  validate_string($env)

  validate_string($statsd_addr)
  validate_string($statsd_type)
  validate_re($ensure, [ '^present$', '^absent$' ])

  # there is an expectation that hostname and pop can be extracted from the
  # value of this parameter
  $node = $::pqdn

  $statsd_prefix = "video.${statsd_type}.${env}"

  $process_name = $backup ? {
    false => 'primary',
    true  => 'secondary',
  }

  $options = prefix(delete_undef_values({
    'private-addr'       => $private_addr,
    'public-addr'        => $public_addr,
    'statsd-addr'        => $statsd_addr,
    'statsd-prefix'      => $statsd_prefix,
    'statsd-rate'        => $statsd_rate,
    'node'               => $node,
    'environment'        => $env,
    'process'            => $process_name,
    'redis-protocol'     => $redis_protocol,
    'redis-address'      => $redis_address,
    'redis-cluster'      => $redis_cluster,
    'upstream-url'       => $upstream_url,
    'fring-url'          => $weaver::ads_fring,
    'ehrmantraut-sqs'    => $weaver::ads_ehrmantraut,
    'keys-url-prefix'    => $weaver::usher_url,
    'beacon-url'         => $beacon_url,
    'enable-longpoll'    => $enable_longpoll,
    'gatekeeper-library' => $gatekeeper_library,
    'gatekeeper-name'    => $gatekeeper_name,
    'gatekeeper-fqdn'    => $gatekeeper_fqdn,
    'ad-concurrency-limit'    => $ad_concurrency_limit,
  }), '-')

  ::daemontools::supervise { $name:
    ensure         => $ensure,
    syslog         => 'local3',
    wd             => "${weaver::basedir}/weaver",
    user           => 'weaver',
    enable_cgroups => true,
    cgroups        => ['memory'],
    cgroups_rules  => {
      'memory' => {
        'memory.swappiness' => 1,
        'memory.limit_in_bytes' => -1,
      }
    },
    predaemon => [
      "mkdir -p /var/run/edgeevent && chmod -R 777 /var/run/edgeevent",
      "test -f /etc/profile.d/proxy.sh && source /etc/profile.d/proxy.sh",
      "test -f /etc/default/weaver && source /etc/default/weaver",
      "test -f /etc/milliner-client-env.sh && source /etc/milliner-client-env.sh",
      "export GOGC=${weaver::gogc}",
      "export GODEBUG=${weaver::godebug}",
      'exec 2>&1',
    ],
    daemon    => 'current/bin/weaver',
    options   => join(join_keys_to_values($options, '='), ' '),
  }

  $public_split_addr = split($public_addr, ':')
  $public_port = $public_split_addr[1]

  $private_split_addr = split($private_addr, ':')
  $private_port = $private_split_addr[1]

  # We use primary/backup here even though it SHOULD be primary/secondary according to latin.
  $instance_tags = $backup ? {
    true  => concat($consul_tags, ['backup', "private-port=${private_port}"]),
    false => concat($consul_tags, ['primary', "private-port=${private_port}"]),
  }


  ::consul::service { $name:
    ensure       => $ensure,
    service_name => 'weaver',
    port         => $public_port,
    consul_tags  => $instance_tags,
    require      => Daemontools::Supervise[$name],
  }
}
