import base64
import datetime
import json
import random
import time
import urllib

from httplib import HTTPConnection

from logster.timeout import timeout


class SpadeException(Exception):
    pass


class SpadeClient(object):
    """Client to write data to Spade.

    host should be the hostname to send data to.

    pct should be between 0.0 and 1.0. This fraction of data will get
    sent to spade (so pct=0.0 means send nothing; pct=1.0 means send
    everything).

    timeout should be an integer. The client will abort sending data
    if it ever takes longer than timeout seconds.

    """
    def __init__(self, host, pct=1.0, timeout=30):
        self.host = host
        self.pct = pct
        self.timeout = timeout

        self.conn = None

        # Use /dev/urandom so we don't need to think about seeds.
        self._rng = random.SystemRandom()

    def connect(self):
        if self.conn is not None:
            return
        try:
            self.conn = HTTPConnection(self.host, timeout=self.timeout)
        except Exception, e:
            raise SpadeException("Can't connect to Spade: %s" % e)

    def _send(self, latreps):
        body = json.dumps([x.to_spade_payload() for x in latreps])
        encoded = urllib.urlencode({"data": base64.b64encode(body)})
        try:
            self.conn.request("POST", "/track", encoded)
        except Exception, e:
            raise SpadeException("Can't send data to Spade: %s" % e)

    def _sample(self, latreps):
        """ randomly select self.pct fraction of the items in latreps """
        # TODO: Implement some sort of consistent algorithm, like
        # hashing the segment address, in order to coordinate which
        # segments get tracked on both this system and the video
        # player client.
        return [x for x in latreps if self._rng.random() < self.pct]

    def send(self, latreps):
        latreps = self._sample(latreps)

        # Chunk into blocks of 100 LatencyReports per POST
        n = 100

        n_sent = 0
        # Use a lambda for the message so that it evaluates n_sent
        # only *after* the timeout occurs.
        msg = lambda: "Timed out ({0} / {1} LatencyReports sent)".format(n_sent, len(latreps))
        with timeout(self.timeout, msg):
            for i in xrange(0, len(latreps), n):
                chunk = latreps[i:i+n]
                self._send(chunk)
                n_sent += len(chunk)

    def close(self):
        self.conn.close()
        self.conn = None
