from logster.logster_helper import MetricObject
import time
import json

class ReplicationTimestamp(object):
    """ReplicationTimestamp holds data describing the replication times
    of a particular segment of video within a particular datacenter.

    The expected use for this is to hold parsed data coming from
    Varnish's logs.

    """
    def __init__(self, dc, start, end):
        self.dc = dc
        self.start = start
        self.end = end

    def _to_dict(self):
        return {
            "dc": self.dc,
            "start": self.start,
            "end": self.end,
        }

    @classmethod
    def from_string(cls, raw):
        """ parse colon-delimited string into a timestamp """
        parts = raw.split(':')

        if len(parts) != 3:
            return None

        dc = parts[0]
        try:
            start = epoch_string_to_float(parts[1])
            end = epoch_string_to_float(parts[2])
        except ValueError:
            return None

        return ReplicationTimestamp(dc, start, end)


    def __repr__(self):
        return "ReplicationTimestamp({dc}, {start}, {end})".format(**self.__dict__)


    def __eq__(self, other):
        if not isinstance(other, ReplicationTimestamp):
            return False
        return (self.dc == other.dc and
                self.start == other.start and
                self.end == other.end)


def latency_reports_to_metric(reports):
    """ Convert a list of LatencyReports into a single MetricObject """
    return MetricObject(
        name="latency_report_list",
        value=reports,
        type=LatencyReport,
    )


class LatencyReport(object):
    """LatencyReport bundles multiple ReplicationTimestamps for the same
    segment into a report to be output to metrics.

    """

    def __init__(self, channel, format, segment, timestamps):
        self.channel = channel
        self.format = format
        self.segment = segment
        self.timestamps = timestamps

    def hash_key(self):
        return "{0},{1},{2}".format(self.channel, self.format, self.segment)

    @classmethod
    def parse_filename(cls, filename):
        # File format looks like
        # '/hls/<channnel>_<id1>_<id2>/<format>/<segment>'. We need to
        # trim out <id1> and <id2>. Also, we need to be prepared for
        # absolute (instead of relative) requests, like
        # 'http://video-edge-9142e8.cdg01.hls.ttvnw.net/hls/<channnel>_<id1>_<id2>/<format>/<segment>'
        parts = filename.rsplit("/", 3)
        if len(parts) < 3:
            return None
        segment = parts[-1]
        format = parts[-2]
        channel = parts[-3]

        # Trim out the underscore-delimited ID fields. There are two
        # of them.
        channel = channel[:channel.rfind("_")]
        channel = channel[:channel.rfind("_")]

        return channel, format, segment

    @classmethod
    def from_string(cls, filename, raw):
        """ parse comma-delimited timestamps from a string """
        if raw is None:
            return None

        parsed_filename = cls.parse_filename(filename)
        if parsed_filename is None:
            return None
        channel, format, segment = parsed_filename

        timestamps = [ReplicationTimestamp.from_string(x) for x in raw.split(',')]
        # LatencyTimestamp.from_string can return None if it
        # encounters an error. Filter out those Nones.
        timestamps = [x for x in timestamps if x is not None]

        if len(timestamps) == 0:
            return None

        return LatencyReport(channel, format, segment, timestamps)

    def to_spade_payload(self):
        dc_count = 0
        min_time = None
        max_time = None
        last_dc = ""
        path = ""
        json_timestamps = []
        for t in self.timestamps:
            dc_count += 1
            if min_time is None or t.start < min_time:
                min_time = t.start
            if max_time is None or t.end > max_time:
                max_time = t.end
            last_dc = t.dc
            path += t.dc + " "
            json_timestamps.append(t._to_dict())

        path = path.strip()
        json_timestamps = json.dumps(json_timestamps)

        return {
            "event": "latency_report_repdist",
            "properties": {
                "segment": self.hash_key(),
                "dc_count": dc_count,
                "min_time": min_time,
                "max_time": max_time,
                "last_dc": last_dc,
                "path": path,
                "timestamps": json_timestamps,
            }
        }

    def __repr__(self):
        return "LatencyReport({channel}, {format}, {segment}, {timestamps})".format(
            channel=self.channel, format=self.format,
            segment=self.segment, timestamps=self.timestamps)

    def __eq__(self, other):
        if not isinstance(other, LatencyReport):
            return False
        return self.segment == other.segment and self.timestamps == other.timestamps


UNIX_NOW = time.time()
SECONDS_PER_YEAR = 365 * 24 * 60 * 60
MIN_VALID_TIMESTAMP = UNIX_NOW - SECONDS_PER_YEAR
MAX_VALID_TIMESTAMP = UNIX_NOW + SECONDS_PER_YEAR
def epoch_string_to_float(raw):
    """Parse a string as a float encoding a unix-epoch timestamp, and
    return the float.

    If the string isn't a float or would indicate a time more than one
    year ago or more than one year in the future, this raises ValueError.

    """
    try:
        v = float(raw)
    except:
        raise ValueError("%s is not parseable as a float" % raw)

    if not MIN_VALID_TIMESTAMP < v < MAX_VALID_TIMESTAMP:
        raise ValueError("not within one year of current time")

    return v
