#!/usr/bin/env python
 
import flvtools.flv as flv
import flvtools.ts_filters as filters
import flvtools.buffered_writter as stream_utils
import optparse
import sys
import time
import signal

p = optparse.OptionParser()
p.add_option('-t', '--tag-window', dest='tag_window', help='Length of window in which timestamps will be analyzed', default=60, type='int')
p.add_option('-w', '--write-buffer', dest='write_buffer', help='Bytes to buffer before writing to output stream', default=16384, type='int')
p.add_option('-x', '--probe-shift', dest='probe_dts_shift', help='Probe timestamps shift amount for a number of tags.', action='store_true')
p.add_option('-s', '--static-shift', dest='static_dts_shift', help='Shift timestamps by a static amount.', type='int')
p.add_option('-c', '--continuous-shift', dest='continuous_dts_shift', help='Continuously detect shift amount.', action='store_true')
p.add_option('-p', '--print-ts', dest='print_ts', help='Only print timestamps.', action='store_true')

options, args = p.parse_args()

if len(args) < 2 and not (options.print_ts and len(args) == 1):
    print "Missing input or output file name."
    sys.exit(1)

if len(args) > 2:
    print "Too many parameters."
    sys.exit(1)

in_filename = args[0]
if in_filename == '-':
    in_file = sys.stdin
else:
    in_file = open(in_filename, 'r')

if options.print_ts:
    flv.print_timestamps(in_file)
    sys.exit(0)

out_filename = args[1]
if out_filename == '-':
    out_file = sys.stdout
else:
    out_file = open(out_filename, 'w')

# Build filter chain
ts_chain = filters.TSNormalizationChain(options.tag_window)

input_gap_detector = filters.GapDetector("Input")
ts_chain.push_input_filter(input_gap_detector)

if options.probe_dts_shift:
    sys.stderr.write("Using CTS probing...\n")
    shift_detector = filters.DetectCTSShiftFilter(options.tag_window)
    ts_chain.push_probe_filter(shift_detector)
elif options.continuous_dts_shift:
    sys.stderr.write("Using continuous CTS shift detection...\n")
    shift_detector = filters.DetectCTSShiftFilter(filters.DetectCTSShiftFilter.CONTINUOUS_SEEK)
    ts_chain.push_input_filter(shift_detector)
elif options.static_dts_shift:
    sys.stderr.write("Constant CTS shift...\n")
    shift_detector = filters.DummyShiftFilter(options.static_dts_shift)
    ts_chain.push_input_filter(shift_detector)
else:
    sys.stderr.write("No shift detection algorithm selected.\n")
    sys.exit(0)

ts_chain.push_input_filter(filters.ShiftCTSFilter(shift_detector))
ts_chain.push_input_filter(filters.ShiftDTSFilter(shift_detector, tag_type = flv.FLV_AUDIO_TAG))
reorder_filter = filters.OrderByDTSFilter()
ts_chain.push_chain_filter(reorder_filter)

output_gap_detector = filters.GapDetector("Output")
ts_chain.push_output_filter(output_gap_detector)

buf_stream = stream_utils.BufferedWriter(out_file, options.write_buffer)
buf_stream.write(flv.writeHeader())

def finish():
    buf_stream.flush()
    end = time.time()

    sys.stderr.write("Done. %s tags, %s reorders, %s front cb inserts (largest %s), final shift %s.\n" % (ts_chain.num_tags, reorder_filter.reorders, reorder_filter.front_inserts, reorder_filter.largest_front_insert, shift_detector.shift_amount()))
    sys.stderr.write("Largest input ts gap: %s @ DTS %s\n" % (input_gap_detector.max_gap, input_gap_detector.max_gap_dts))
    sys.stderr.write("Largest output ts gap: %s @ DTS %s\n" % (output_gap_detector.max_gap, output_gap_detector.max_gap_dts))
    sys.stderr.write('Elapsed time: ' + `time.time() - start` + '\n')
    sys.exit(0)

signal.signal(signal.SIGHUP, finish)
signal.signal(signal.SIGINT, finish)
signal.signal(signal.SIGTERM, finish)

# TODO: should arguably be in TSNormalizationChain.normalize()
tag_gen = flv.readFile(in_file, chunk_size = 1024, avc_info = True)

start = time.time()
for tag in ts_chain.normalize(tag_gen):
    buf_stream.write(filters.serialize_tag(tag))

finish()
