#!/usr/bin/env python
from twisted.internet import reactor, defer
from twisted.python import log, usage
from flvtools import flv
from flvtools.factories import rtmp
import time, sys, os, httplib, struct, traceback

options = {}


def fatal_error(reason):
    log.err(reason)
    log.err("Stopping reactor")
    reactor.stop()
    
class fake_thing:
    def get_uptime(*args):
        return 1
    def plug(command, *args, **kwargs):
        log.msg("Command: %s"%command)
        log.msg(repr(args))
        log.msg(repr(kwargs))
        if command == 'invoke':
            log.msg('Got an invoke!!!')
            return True

class FlvPusher:
    def __init__(self):
        self.stream_id = 4
        self.channel_id = 4
        self.broadcast_stream_id = 1.0
        self.in_file = None
        self.stream_key = None
        self.core = None
        self.client = None
        
        self.push_buffer = []  
        self.read_done = False
        self.fill_buffer = True
        
        #first tag in stream and time it was read, used to time reads
        self.first_tag = None
        self.first_tag_time = None
        
        #some tag and the time it was written (used to time writes; must be reset every time that the write rate changes)
        self.out_reference_tag = None
        self.out_reference_tag_time = None 
        
        self.last_write_time_stamp = None
        self.cur_scaler = -1 
        
        self.connected = False 
        
    def readLoop(self, tag_gen, index=0):    
        try:
            start_mark = time.time()
            tag = tag_gen.next()
            if not tag:
                #need to wait for data
                reactor.callLater(0.2, self.readLoop, tag_gen, index)
                return 
        
            self.push_buffer.append(tag)
            
            if self.first_tag is None:
                self.first_tag = tag
                self.first_tag_time = time.time()
                
            if options['read-scale'] >= 0:
                elap_time = time.time() - self.first_tag_time
                elap_tag_time = tag['time_stamp'] - self.first_tag['time_stamp']
                to_sleep = (elap_tag_time/1000.0)*options['read-scale'] - elap_time
                if to_sleep < 0: 
                    to_sleep = 0
            else:
                to_sleep = 0.0005
                    
            reactor.callLater(to_sleep, self.readLoop, tag_gen, index+1)
            return 
        except StopIteration:
            self.read_done = True
            log.msg("read done") 

        
    def writeLoop(self, index=0):
        start_mark = time.time()
        
        try:
            connection = self.client.get_nc()
            if not connection:
                fatal_error("push connection closed")
                return
            if (self.fill_buffer):    
                to_sleep = 0.2
                if (self.push_buffer and self.push_buffer[-1]['time_stamp'] - self.push_buffer[0]['time_stamp'] >= options['buffer']) or self.read_done:
                    self.fill_buffer = False
                    #reset out time reference
                    self.out_reference_tag =  self.push_buffer[0]
                    self.out_reference_tag_time = start_mark
                    to_sleep = 0
                    if self.push_buffer:
                        log.msg("Buffer full (with length %s)"%(self.push_buffer[-1]['time_stamp'] - self.push_buffer[0]['time_stamp']))
                        
                reactor.callLater(to_sleep, self.writeLoop, index)
                return
            else:
                to_sleep = 0.1
                if self.push_buffer:
                    if self.out_reference_tag is None:
                        self.out_reference_tag = self.push_buffer[0]
                        self.out_reference_tag_time = start_mark
                    elap_time = (start_mark - self.out_reference_tag_time)*1000 
                    
                    while(self.push_buffer and (self.push_buffer[0]['time_stamp'] - self.out_reference_tag['time_stamp'])*self.cur_scaler < elap_time):
                        to_sleep = 0.0005
                        t = self.push_buffer.pop(0)
                        data, type, timestamp = t["data"], t["type"], t["time_stamp"]
                        if not self.last_write_time_stamp is None:
                            ts_diff = timestamp - self.last_write_time_stamp
                        else:
                            ts_diff = 0
                        
                        self.last_write_time_stamp = timestamp
                        index += 1
                        log.msg("Pushing %s bytes of type %s and absolute timestamp %s in tag # %s" % (len(data), type, timestamp, index))
                        data = connection.encoder.process(data, self.channel_id, self.broadcast_stream_id, type, ts_diff)
                        connection.write(data)    
                    
                    if self.push_buffer:
                        old_scaler = self.cur_scaler
                        buffer_length = self.push_buffer[-1]['time_stamp'] - self.push_buffer[0]['time_stamp']
                        log.msg("buffer_length: %s" % buffer_length)
                        if buffer_length > options['buffer']*1.2:
                            self.cur_scaler = options['scale'] * 0.995 
                        elif buffer_length < options['buffer']/1.2:
                            self.cur_scaler = options['scale'] * 1.005
                        if self.cur_scaler != old_scaler:
                            #reset reference because playback rate has changed 
                            self.out_reference_tag = self.push_buffer[0]
                            self.out_reference_tag_time = start_mark
                                        
                else:
                    if self.read_done:
                        log.msg("push done")
                        reactor.stop()
                        return
                    self.fill_buffer = True
                    log.msg("Buffering...")
                    to_sleep = 0.2
                    
                reactor.callLater(to_sleep, self.writeLoop, index)
        except Exception:
            log.err(traceback.format_exc())
            reactor.stop()                     
                
    @defer.inlineCallbacks
    def push(self):
        reactor.callLater(float(options['connect_timeout']), self.check_timeout)
        self.stream_key = options['stream']
        self.core = fake_thing()
        self.core.config = {'max_conn_per_ip': 666, 'max_conn': 666}
        self.core.name = "fake"
        self.core.nodes = {self.core.name: fake_thing()}
        #client = rtmp.RTMPServer(core, client=True, app=options['app'], tcUrl = "rtmp://%s/%s"%(options['host'], options['app']), swfUrl=options['swfurl'])
        params = {'flashVer' : 'PMS'}
        self.client = rtmp.RTMPServer(self.core, 
            client=True, 
            app=options['app'], 
            tcUrl = "rtmp://%s/%s" % (options['host'],options['app']), 
            swfUrl = options['swfurl'], 
            params=params )
        reactor.connectTCP(options['host'], int(options['port']), self.client)
        log.msg("Connecting...")
        args = yield self.client.connectDeferred
        log.msg("CONNECTED! args = %s" % args)
        self.client.invoke('createStream', self.stream_id)
        yield self.client.createStreamDeferred
        self.client.invoke('publish', 0.0, [self.stream_key, 'live'], streamId=self.broadcast_stream_id)
        args = yield self.client.publishDeferred
        print "PUBLISHING to %s - args = %s" % (self.stream_key, args)
        tag_gen = flv.readFileNoBlock(self.in_file.fileno(), chunk_size=512)
        self.start_time = time.time()
        reactor.callLater(0, self.readLoop, tag_gen)
        reactor.callLater(0, self.writeLoop, 0)
        
        self.connected = True
        
        defer.returnValue(0)
        return

    def check_timeout(self):
        if not self.connected:
            fatal_error("timeout")
            
         
def getStream(credentials):
    username, password = credentials.split(":", 1)
    conn = httplib.HTTPConnection('live.justin.tv')
    conn.request("GET", '/stream_key.csv?username=%s&password=%s' % (username, password))
    r = conn.getresponse()
    page = r.read()
    if page.find("BAD_AUTHORIZATION") > 0:
        print ("ERROR - Could not authenticate with Justin.TV using the credentials you gave (username='%s', password='%s')."
               % (username, password))
        os._exit(-1)
    else:
        key = page.split("\n")[1].strip()
        log.msg("STREAM KEY FOUND: '%s'" % key)
        return key

def check_int(v):
    try:
        val = int(v)
    except Exception:
        raise ValueError("Not an integer")
    return val     

def check_float(v):
    try:
        val = float(v)
    except Exception:
        raise ValueError("Not a float")
    return val     


if __name__ == "__main__":
    class Options(usage.Options):
        optFlags = [
            ['verbose', 'v', 'Detailed logging'],
            ['akamai', 'k', 'Run in akamai mode'],
            ['rush', 'r', '-x 0'],
        ]
        optParameters = [
            ['file', 'f', '', 'Input file name'],
            ['credentials', 'c', 'None', 'username:password (from Justin.TV)'],
            ['stream', 's', 'test', 'Stream name'],
            ['host', 'h', '127.0.0.1', 'RTMP server'],
            ['port', 'p', 1935, 'RTMP port', check_int],
            ['log', 'l', 'stdout', 'Log file'],
            ['swfurl', 'w', 'http://www.justin.tv/test.swf', 'swfUrl'],            
            ['app', 'a', 'app', 'Rtmp application'],
            ['buffer', 'b', 0, 'Length of buffer (in milliseconds)', check_float],
            ['scale', 'x', 1, 'Scale replay rate by this factor', check_float],
            ['read-scale', 'X', -1, 'Scale read rate by this factor (-1 = unlimited)', check_float],
            ['connect_timeout', 't', 5, "Timeout on connection (seconds)"],
        ]
    options = {}
    try:
        options = Options()
        options.parseOptions()
    except usage.UsageError, errortext:
        print '%s: %s' % (sys.argv[0], errortext)
        print '%s: Try --help for usage details.' % (sys.argv[0])
        sys.stdout.flush()
        sys.exit(1)
    
    pusher = FlvPusher()
     
    if options['file']:
        pusher.in_file = open(options['file'], 'r')
    else:
        pusher.in_file = sys.stdin
            
    if options['verbose']:
        if options['log'] == 'stdout':
            l = sys.stdout
        else:
            l = open(options['log'], 'w')
        log.startLogging(l)
    
    if options['rush']:
        options['scale'] = 0
        
    pusher.cur_scaler = options['scale']
    
    if options['akamai']:
        pusher.broadcast_stream_id = pusher.stream_id
    
    if options['credentials'].find(":") > 0:
        options['stream'] = getStream(options['credentials'])
    
    if options['buffer']:
        pusher.fill_buffer = True
    print options['buffer'] 
       
    pusher.push().addErrback(fatal_error)
    reactor.run()
