"""
    
    File: util.py
    Description:
        
        Class to limit connection to a given service.
    
    Author: Kyle Vogt
    Date  : January 18th, 2008
    Copyright (c) 2008, Justin.tv, Inc.

"""

from twisted.python import log
import time
import sys
import random
from twisted.internet import reactor, defer

class BackoffRetry(object):
    def __init__(self, delay=10, max_delay=90, max_retry=3, backoff=2, jitter=.15):
        self.delay = delay
        self.backoff = backoff
        self.max_delay = max_delay
        self.jitter = jitter
        self.curr_delay = delay
        self.max_retry = max_retry
        self.tries = 0

    def __call__(self, fn):
        def retry(*args):
            if not fn(*args):
                self.curr_delay *= 1 - random.uniform(0, self.jitter)
                log.msg("Retrying in %s seconds" % self.curr_delay)

                self.tries += 1
                if self.tries >= self.max_retry:
                    self.tries = 0
                    return

                reactor.callLater(self.curr_delay, retry, *args)
                self.curr_delay *= self.backoff
                if self.curr_delay > self.max_delay:
                    self.curr_delay = self.max_delay
                return

            self.curr_delay = self.delay
            self.tries = 0

        return retry

class ConnectionManager:
    
    def __init__(self, ip_limit = 50, limit = 5000):
        "Allow override of connections per ip and total connections"
        self.clients = []
        self.hosts = {}
        self.ip_connection_limit = ip_limit
        self.connection_limit = limit
    
    def add(self, client, logConnection=True):
        "Add a new client if it doesn't exceed connection limits"
        conns = self.hosts.get(client.ip, 0)
        if conns >= self.ip_connection_limit and client.ip != '127.0.0.1':
            log.msg("Connection rejected, at ip connection limit of %s" % self.ip_connection_limit)
            return False
        elif len(self.clients) + 1 > self.connection_limit:
            log.msg("Connection rejected, at global connection limit of %s" % self.connection_limit)
            return False
        else:
            self.hosts[client.ip] = conns + 1
            self.clients.append(client)
            if logConnection:
                log.msg("Connection accepted. %s has %s connections. %s total" % (client.ip, self.hosts[client.ip], len(self.clients)))
            return True
    
    def remove(self, client):
        if client in self.clients:
            conns = self.hosts.get(client.ip, 0)
            if conns > 0: 
                conns -= 1
                self.hosts[client.ip] = conns
            else:
                del self.hosts[client.ip]
            self.clients.remove(client)
            return True
        else: return False

class Profiler:
    
    def __init__(self):
        self.time_start = {}
        self.time_used = {}
        self.time_length = 10.0
    
    def benchmark(self, func, *kw, **kwargs):
        if func not in self.time_start:
            self.reset(func)
        
        st = time.time()
        func(*kw, **kwargs)
        self.time_used[func] += (time.time() - st)
        if (time.time() - self.time_start[func]) > self.time_length:
            log.msg('Used %.3fs out of 10s for func %s' % (self.time_used[func], func))
            self.reset(func)
    
    def reset(self, func):
        self.time_start[func] = time.time()
        self.time_used[func] = 0

        
