"""

    File: netstream.py
    Description: 
    
        Module to implement a NetStream-like object

    Author: Kyle Vogt
    Date  : October 15th, 2007
    Copyright (c) 2007, Justin.tv, Inc.
    
"""
from twisted.python import log
from twisted.internet import reactor, defer
from flvtools.meta import Stream
from flvtools.rtmp import amf_py as amf
from flvtools.rtmp import basicnetstream
from flvtools import flv
from flvtools import messages
import struct, time, traceback, re

class NetStream(basicnetstream.BasicNetStream):
        
    def __repr__(self):
        return '<NetStream with id %s>' % self.streamId
    
    def publish(self, name):
        "Create a stream buffer and record to it"
        if name in self.core.streams:
            origin = self.core.streams[name].origin
            obj = { 'level' : 'status',
                    'code' : 'NetStream.Publish.BadName',
                    'description' : 'PMS already has stream %s (from node %s)' % (name, origin)}
            log.msg(obj['description'])
            self.nc.invoke('onStatus', 0.0, obj)
            self.nc.shutdown()
        #if (not self.core.nodes[self.core.name].active                                   or
        #    float(self.core.load) > float(self.core.config.get('server_max_load', 0.99)) ):
        #    obj = { 'level' : 'status',
        #            'code' : 'NetStream.Publish.ServerFull',
        #            'description' : 'This server cannot accept any more streams, please try another.' }
        #    log.msg(obj['description'])
        #    self.nc.invoke('onStatus', 0.0, obj)
        #    self.nc.shutdown()
        elif isinstance(name, str):
            # Create stream object
            me = self.core.nodes[self.core.name]
            self.stream = Stream(
                name=name,
                origin=self.core.name,
                nodes=[':'.join([me.public_ip, me.rtmp_port])],
                node_names=[me.name],
                start=time.time(),
                local=True,
                clip=None,
                id=self.streamId,
                core=self.core)
            log.msg('Running authentication plugins')
            result = defer.maybeDeferred(self.core.plug, 'authenticate', self.nc.ip, name)
            result.addCallbacks(
                callback = self.authenticatedPublish, callbackArgs = [self.nc.ip, name],
                errback = self.authFailure, errbackArgs = [self.nc.ip, name])
        else:
            log.msg("Bad stream name, ignoring: %s" % name)

    def authenticatedPublish(self, results, host, original_name):
        "Auth plugin reported success or failure"
        try:
            # Valid data?
            if not results:
                self.authFailure(results, host, original_name)
                return
            if 'stream' not in results:
                log.msg("Something went wrong unpacking results %s" % results)
                self.authFailure(results, host, original_name)

            obj = { 'level' : 'status', 
                    'code' : 'NetStream.Publish.Start',
                    'description' : 'Publishing %s.' % results['stream']}
            log.msg(obj['description'])
            self.nc.invoke('onStatus', 0.0, obj, streamId=self.streamId)

            # Read JTV user ID
            try: uid = original_name.split('_')[1]
            except: uid = None
            self.stream.user_id = uid
            
            # Handle replications
            self.stream.name = results['stream']
            if results.get('type') == 'replication' and results['stream'] in self.core.streams:
                real_stream = self.core.streams[results['stream']]
                self.stream.replicated = True
                self.stream.origin = real_stream.origin
                self.stream.password = real_stream.password
                self.stream.embed_enabled = real_stream.embed_enabled
            else:
                self.stream.password = results.get('password', None)
                self.stream.embed_enabled = results.get('embed_enabled', True)

            # Connect stream to plugins
            if 'process' in self.core.plugCache:
                for obj in self.core.plugCache['process']:
                    self.stream.subscribe(obj)

            # Set up new stream
            self.core.streams[results['stream']] = self.stream
            self.core.update(self.stream, messages.UP)
            self.publishing = True
            self.nc.publishing += 1
        except:
            log.msg('Error publishing:')
            log.err()

    def authFailure(self, result, host, original_name):
        "Auth failed."
        obj = { 'level' : 'error',
                'code' : 'NetStream.Publish.Rejected',
                'description' : 'Authentication failed.'}
        log.msg(obj['description'])
        log.msg('Stream name was %s' % original_name)
        self.nc.invoke('onStatus', 0.0, obj, streamId=self.streamId)
        self.close()            
                    
    def stop(self):
        "Stop playing or publishing"
        if self.playing and self.stream:
            self.stream.unsubscribe(self)
            self.stream.removeViewer(self.nc.ip)
            self.core.update_counts()
            self.playing = False
            self.nc.playing -= 1
            # Log this event
            try:
                log.msg(stat='PLAY', data={
                    'ip' : self.nc.ip,
                    'ip_conns' : self.nc.factory.conn_manager.hosts.get(self.nc.ip, 0),
                    'stream' : self.stream.name,
                    'length' : int(time.time() - self.playTime),
                    'bytes' : self.nc.bytesSent - self.byteOffset,
                    'node' : self.nc.factory.core.name,
                    'tcUrl' : self.nc.tcUrl,
                    'pageUrl' : self.nc.pageUrl,
                    'play_data' : self.play_data,
                })            
            except:
                log.msg('Error writing play log')
                log.err()
        if self.publishing and self.stream:
            self.publishing = False
            self.nc.publishing -= 1
            self.stream.reason = 'Stream stopped.'
            # Log this event
            try:
                log.msg(stat='PUBLISH', data={
                    'ip' : self.nc.ip,
                    'ip_conns' : self.nc.factory.conn_manager.hosts.get(self.nc.ip, 0),
                    'stream' : self.stream.name,
                    'length' : int(time.time() - self.stream.start),
                    'bytes' : self.nc.bytesSent - self.byteOffset,
                    'node' : self.nc.factory.core.name,
                    'replicated' : self.stream.replicated,
                    'tcUrl' : self.nc.tcUrl,
                    'pageUrl' : self.nc.pageUrl,
                })
            except:
                log.msg('Error writing publish log')
                log.err()
            # Shut down subscribers
            self.stream.shutdown()
            self.core.update_counts()
            # Tell other nodes the stream is down
            self.core.update(self.stream, messages.DOWN)        
                
    def process(self, stream, packets, ignoreId=False, stop=False):
        "Send the packets"
        if not self.running: return
        if stop:
            obj = { 'level' : 'status',
                    'code' : 'NetStream.Publish.Stop',
                    'description' : 'Stopped publishing stream %s.' % stream.name}
            #log.msg(obj['description'])
            self.nc.invoke('onStatus', 0.0, obj, streamId=self.streamId)
            return
        for packet in packets:                
            if packet['absflag'] or ignoreId: 
                tc = packet['abstime']
                absflag = True
            elif packet['object'] not in self.objects:
                tc = packet['abstime']
                absflag = True
                #log.msg("Forced absolute timestamp for objectId %s" % packet['object'])
                self.objects.append(packet['object'])
            else:
                tc = packet['timecode']
                absflag = False
            if self.waitingVP6 and packet['key']: 
                self.waitingVP6 = False
                self.sendStart()
            elif self.waitingSOR and packet['key']:
                self.waitingSOR = False
                self.sendFirstKey()    
                self.sendStart()
            # Make sure to send this keyframe now that we're not waiting!
            if not self.waitingVP6:
                data = self.nc.encoder.process(packet['data'], packet['object'], self.streamId, packet['type'], tc, absflag)
                #self.nc.write(data)
                self.chunks.append(data)
                self.chunkBytes += len(data)
        
        #st = time.time()
        while not self.nc.dropper.paused and len(self.chunks):
            data = self.chunks.pop(0)
            self.nc.write(data)
            self.chunkBytes -= len(data)
        if self.chunkBytes > 500000:
            self.chunkBytes = 0
            self.chunks = []
            obj = {    'level' : 'status', 
                    'code' : 'NetStream.Play.InsufficientBW',
                    'description' : 'Client is downloading too slowly or server is overloaded.'}
            log.msg(obj['description'])
            self.nc.invoke('onStatus', 0.0, obj, streamId=self.streamId)
            self.running = False
            reactor.callLater(3.0, self.nc.shutdown)
        
        #self.elapsed += time.time() - st
        #now = time.time() - self.playTime
        #log.msg('Encoded for %s seconds (in %s actual seconds): %s streams' % (self.elapsed, now, now / self.elapsed))


    def play(self, name):
        "Subscribe to a stream buffer and play it"
        if self.playing: 
            return
        # Start counting bytes for log
        self.byteOffset = self.nc.bytesSent

        # Find the stream
        stream = self.core.streams.get(name)
        available_streams = [s for s in self.core.nodes[self.core.name].streams
                               if s in self.core.streams and self.core.streams[s].local]

        # Does the stream exist?
        if not stream:
            desc = 'Stream %s not found.' % name
            obj = { 'level' : 'status',
                    'code' : 'NetStream.Play.StreamNotFound',
                    'description' : desc}
            log.msg(obj['description'])
            self.nc.invoke('onStatus', 0.0, obj)

        # Is the stream on this server?
        elif stream.name in available_streams:
            # Is there a password and is it correct?
            if self.nc.password and not stream.password:
                obj = { 'level': 'status',
                        'code': 'NetStream.Authenticate.NotRequired',
                        'description': 'This stream has no password' }
                log.msg(obj['description'])
                self.nc.invoke('onStatus', 0.0, obj, channel=amf.EVENT_CHANNEL)
                # don't return, this is ok
            elif stream.password != self.nc.password:
                obj = { 'level' : 'status',
                        'code': 'NetStream.Authenticate.Failed',
                        'description': 'You must supply a valid password by sending NetStream.Authenticate.AccessCode to play %s (provided: %s)' % (name, self.nc.password) }
                log.msg(obj['description'])
                self.nc.invoke('onStatus', 0.0, obj, channel=amf.EVENT_CHANNEL)
                return
            # Is this domain authorized to play the stream?
            if not stream.embed_enabled:
                log.msg('Embed disabled for %s, checking pageUrl: %s' % (name, self.nc.pageUrl))
                if not self.nc.pageUrl or not re.match('(http://)?[a-zA-Z]*.justin.tv', self.nc.pageUrl):
                    obj = { 'level' : 'status',
                            'code' : 'NetStream.Play.StreamNotFound',
                            # TODO: Send the proper code so that we can handle this properly on the client side
                            #'code': 'NetStream.Authenticate.UnauthorizedDomain',
                            'description': 'This domain is not authorized to play the stream %s.' % name}
                    log.msg(obj['description'])
                    self.nc.invoke('onStatus', 0.0, obj, channel=amf.EVENT_CHANNEL)
                    return
            # Do we have too many connections?
            if stream.origin == self.core.name:
                per_stream = int(self.core.config['stream_max_clients_origin'])
            else:
                per_stream = int(self.core.config['stream_max_clients'])
            #if (not self.core.nodes[self.core.name].active                                     or
            #    (float(self.core.load) > float(self.core.config.get('server_max_load', 0.99))) or
            #    (int(stream.counts.get(self.core.name, 0)) > per_stream)                       ): 
            #    obj = { 'level' : 'status',
            #            'code' : 'NetStream.Play.ServerFull',
            #            'description' : 'This server has too many connections, please try another.'}
            #    log.msg(obj['description'])
            #    self.nc.invoke('onStatus', 0.0, obj, channel=amf.EVENT_CHANNEL)
            #    self.nc.shutdown()
            #    return
            # We made it!  Play the stream.
            self.stream = stream
            obj = { 'level' : 'status',
                    'code' : 'NetStream.Play.Reset',
                    'description' : 'Playing and resetting %s.' % name}
            log.msg(obj['description'])
            self.nc.invoke('onStatus', 0.0, obj, channel=amf.EVENT_CHANNEL)
            self.nc.send(amf.META_CHANNEL, amf.PING, 0, chr(0x00) + chr(0x04) + struct.pack("!I", self.streamId))
            self.nc.send(amf.META_CHANNEL, amf.PING, 0, chr(0x00) + chr(0x00) + struct.pack("!I", self.streamId))
            self.core.update_counts()
            self.stream.subscribe(self)
            self.stream.addViewer(self.nc.ip)
            self.playKey()
            self.playing = True
            self.nc.playing += 1
            self.playTime = time.time()

        # Stream is located elsewhere, redirect
        else: 
            desc = 'Stream ' + name + ' found at: ' + ','.join(stream.getMeta().get('nodes', ''))
            obj = { 'level' : 'status',
                    'code' : 'NetStream.Play.StreamNotFound',
                    'description' : desc}
            log.msg(obj['description'])
            self.nc.invoke('onStatus', 0.0, obj)

    def playKey(self):
        "Chose a place in the buffer to start streaming"
        fastStart = self.core.config['fast_start']
        if fastStart and self.stream.last_key and self.stream.video_codec == 'vp6':
            log.msg("Used fastStart")
            self.process(self, [self.stream.last_key], ignoreId=True)
            self.sendFirstKey()
            self.waitingVP6 = True
        else:
            self.waitingSOR = True
            
    def sendStart(self):
        obj = { 'level' : 'status',
                'code' : 'NetStream.Play.Start',
                'description' : 'Started playing %s.' % self.stream.name}
        log.msg(obj['description'])
        self.nc.invoke('onStatus', 0.0, obj, channel=amf.EVENT_CHANNEL)
        # H264 Samples access
        obj = ['|RtmpSampleAccess', False, False]
        self.nc.send(amf.EVENT_CHANNEL, amf.NOTIFY, 0, self.nc.amf.encode(obj))
        # Stream metadata
        if self.stream.meta:
            pass
            self.nc.invoke('onMetaData', self.stream.meta)
        
    def sendStop(self):
        obj = { 'level' : 'status',
                'code' : 'NetStream.Play.Stop',
                'description' : 'Stopped playing %s.' % self.stream.name}
        log.msg(obj['description'])
        self.nc.invoke('onStatus', 0.0, obj, channel=amf.EVENT_CHANNEL)
        
    def sendFirstKey(self):
        obj = { 'level' : 'status',
                'code' : 'NetStream.Play.FirstKey',
                'description' : 'First keyframe sent.'}
        log.msg(obj['description'])
        self.nc.invoke('onStatus', 0.0, obj, channel=amf.EVENT_CHANNEL)
        
    def getStreamIdAndMetaStream(self):
        try:
            return [[i, s.stream] for i, s in self.nc.streams.items() if s.stream][0]
        except:
            return [None, None]
    
