import Component from 'ember-component';
import computed from 'ember-computed';
import injectService from 'ember-service/inject';

export default Component.extend({
  api: injectService(),
  badges: injectService(),
  session: injectService(),

  chatUser: null,

  init() {
    this._super(...arguments);

    let { isAuthenticated, userData } = this.get('session');
    if (!isAuthenticated || !userData) {
      return;
    }

    let options = {
      headers: {
        Authorization: `OAuth ${userData.chat_oauth_token}`
      }
    };
    this.set('options', options);

    let api = this.get('api');
    api.request('get', `/kraken/users/${userData.login}/chat/badges`, '', options).then((result) => {
      if (this.isDestroyed) { return; }
      this.set('availableGlobalBadges', result.available_badges || []);
      this.set('selectedGlobalBadge', result.selected_badge || {});
    });

    let channel = this.get('chatUser.channel');

    if (channel && channel.id) {
      api.request('get', `/kraken/users/${userData.login}/chat/channels/${channel.id}/badges`, "", options).then((result) => {
        if (this.isDestroyed) { return; }
        this.set('availableChannelBadges', result.available_badges || []);
        this.set('selectedChannelBadge', result.selected_badge || {});
      });
    }
  },

  userData: computed.readOnly('session.userData'),

  processBadgeOptions(availableBadges, selectedBadge) {
    let badgeOptions = this.get('badges').getBadgesData(availableBadges);

    for (let i = 0; i < badgeOptions.length; i++) {
      badgeOptions[i].isSelected = (badgeOptions[i].id === selectedBadge.id);
    }

    return badgeOptions;
  },

  noGlobalBadgeSelected: computed('selectedGlobalBadge', function () {
    let selectedGlobalBadge = this.get('selectedGlobalBadge');
    return selectedGlobalBadge === undefined || selectedGlobalBadge.id === undefined;
  }),

  hasAvailableGlobalBadges: computed.notEmpty('availableGlobalBadges'),

  globalBadgeOptions: computed('availableGlobalBadges', 'selectedGlobalBadge', function () {
    return this.processBadgeOptions(this.get('availableGlobalBadges'), this.get('selectedGlobalBadge'));
  }),

  selectedGlobalBadgeOption: computed('selectedGlobalBadge', function () {
    return this.get('badges').getBadgesData([this.get('selectedGlobalBadge')])[0];
  }),

  noChannelBadgeSelected: computed('selectedChannelBadge', function () {
    let selectedChannelBadge = this.get('selectedChannelBadge');
    return selectedChannelBadge === undefined || selectedChannelBadge.id === undefined;
  }),

  hasAvailableChannelBadges: computed.notEmpty('availableChannelBadges'),

  channelBadgeOptions: computed('availableChannelBadges', 'selectedChannelBadge', function () {
    return this.processBadgeOptions(this.get('availableChannelBadges'), this.get('selectedChannelBadge'));
  }),

  selectedChannelBadgeOption: computed('selectedChannelBadge', function () {
    return this.get('badges').getBadgesData([this.get('selectedChannelBadge')])[0];
  }),

  refreshBadgeSet() {
    let userData = this.get('userData');
    let channel = this.get('chatUser.channel');

    this.get('api').request('get', `/kraken/users/${userData.login}/chat/channels/${channel.id}`).then((response) => {
      if (this.isDestroyed) { return; }
      this.get('chatUser.tmiRoom').setBadges(userData.login, response.badges);
    });
  },

  actions: {
    setGlobalBadge(badgeOption) {
      this.set('selectedGlobalBadge', badgeOption);

      let userData = this.get('userData');
      let options = this.get('options');
      this.get('api').request('put', `/kraken/users/${userData.login}/chat/badges?selected_badge=${badgeOption.id}`, '', options).then(() => {
        if (this.isDestroyed) { return; }
        this.refreshBadgeSet();
      });
    },

    removeGlobalBadge() {
      this.set('selectedGlobalBadge', {});

      let userData = this.get('userData');
      let options = this.get('options');
      this.get('api').request('put', `/kraken/users/${userData.login}/chat/badges?selected_badge=`, '', options).then(() => {
        if (this.isDestroyed) { return; }
        this.refreshBadgeSet();
      });
    },

    setChannelBadge(badgeOption) {
      this.set('selectedChannelBadge', badgeOption);

      let userData = this.get('userData');
      let channel = this.get('chatUser.channel');
      let options = this.get('options');
      this.get('api').request('put', `/kraken/users/${userData.login}/chat/channels/${channel.id}/badges?selected_badge=${badgeOption.id}`, '', options).then(() => {
        if (this.isDestroyed) { return; }
        this.refreshBadgeSet();
      });
    },

    removeChannelBadge() {
      this.set('selectedChannelBadge', {});

      let userData = this.get('userData');
      let channel = this.get('chatUser.channel');
      let options = this.get('options');
      this.get('api').request('put', `/kraken/users/${userData.login}/chat/channels/${channel.id}/badges?selected_badge=`, '', options).then(() => {
        if (this.isDestroyed) { return; }
        this.refreshBadgeSet();
      });
    }
  }

});
