/* globals i18n */

import Component from 'ember-component';
import injectService from 'ember-service/inject';
import computed from 'ember-computed';
import { htmlSafe } from 'ember-string';

const NOT_DELAYED_MESSAGE = i18n('The stream is not currently delayed');
const DELAYED_MESSAGE_MINUTE = i18n('The stream is delayed by 1 minute');
const DELAYED_MESSAGE_MINUTES = i18n('The stream is delayed by {{amount}} minutes');
const DELAY_BUTTON_LABEL = i18n('Delay ({{delay}}min)');

export default Component.extend({
  ajax: injectService(),

  classNames: ['js-dashboard-stream-delay'],

  /* passed in properties */
  channel: null,
  isLive: null,
  trackDashboardEvent() {},

  /* internal properties */
  delayLoading: false, // set to true in production
  delayError: false,
  delay: null,
  prevDelay: null,
  delayMessage: NOT_DELAYED_MESSAGE,

  delayButtonLabel: computed('delay', function() {
    let label = DELAY_BUTTON_LABEL;
    let delay = this.get('delay');

    return htmlSafe(label.replace('{{delay}}', delay));
  }),

  isDelayDisabled: computed.or('delayLoading', 'isLive'),

  init() {
    this._super(...arguments);

    let delay = this.get('channel.delay') / 60;

    if (isNaN(delay)) {
      delay = 0;
    }

    this.set('delay', delay);
    this.set('prevDelay', delay);

    if (delay === 1) {
      this.set('delayMessage', DELAYED_MESSAGE_MINUTE);
    } else if (delay > 1) {
      let message = DELAYED_MESSAGE_MINUTES;
      this.set('delayMessage', message.replace('{{amount}}', delay));
    }
  },

  actions: {
    delayChanged(value) {
      let prevDelay = this.get('delay');
      let trackDashboardEvent = this.get('trackDashboardEvent');

      trackDashboardEvent('live_dashboard_stream_delay', {
        action: 'slidermove',
        slider_start_position: prevDelay,
        slider_position: parseInt(value)
      });

      this.set('prevDelay', prevDelay);
      this.set('delay', parseInt(value));
    },

    removeError() {
      this.set('delayError', false);
      this.set('delayLoading', false);
    },

    updateDelay() {
      let delay = this.get('delay');
      let channelId = this.get('channel.id');
      let delayUrl = `/${ channelId }/dashboard/delay`;
      let isUpdateSuccessful = null;
      let options = {
        dataType: 'text',
        data: { delay: delay }
      };

      this.set('delayError', false);
      this.set('delayLoading', true);

      this.get('ajax').post(delayUrl, options)
        .then(() => {
          if (this.isDestroyed) { return; }

          isUpdateSuccessful = true;

          if (delay === 0) {
            this.set('delayMessage', NOT_DELAYED_MESSAGE);
          } else if (delay === 1) {
            this.set('delayMessage', DELAYED_MESSAGE_MINUTE);
          } else {
            let message = DELAYED_MESSAGE_MINUTES;
            this.set('delayMessage', message.replace('{{amount}}', delay));
          }
        }, () => {
          if (this.isDestroyed) { return; }

          isUpdateSuccessful = false;

          let oldDelay = this.get('channel.delay') / 60;

          if (isNaN(oldDelay)) {
            oldDelay = 0;
          }

          this.set('delay', oldDelay);

          if (oldDelay === 1) {
            this.set('delayMessage', DELAYED_MESSAGE_MINUTE);
          } else if (oldDelay > 1) {
            let message = DELAYED_MESSAGE_MINUTES;
            this.set('delayMessage', message.replace('{{amount}}', oldDelay));
          }

          this.set('delayError', true);
        })
        .finally(() => {
          if (this.isDestroyed) { return; }

          let trackDashboardEvent = this.get('trackDashboardEvent');

          trackDashboardEvent('live_dashboard_stream_delay', {
            action: 'streamdelayclick',
            slider_start_position: this.get('prevDelay'),
            slider_position: delay,
            update_successful: isUpdateSuccessful
          });

          this.set('delayLoading', false);
        });
    }
  }
});
