import Component from 'ember-component';
import observer from 'ember-metal/observer';
import computed from 'ember-computed';
import $ from 'jquery';
import moment from 'moment';
import run from 'ember-runloop';

import { DAYS_RESOLUTION, DATE_FORMAT } from 'web-client/components/dashboards/revenue/date-picker/component';

export default Component.extend({
  classNames: ['flex__item'],

  // Passed in properties:
  momentStartDate: null,
  momentEndDate: null,
  isDatepickerDisabled: null,
  updateDateFilter: null,

  // Own properties:
  dayDifference: computed('momentStartDate', 'momentEndDate', function() {
    return this.get('momentEndDate').diff(this.get('momentStartDate'), DAYS_RESOLUTION) + 1;
  }),

  didInsertElement() {
    this._super(...arguments);

    let momentStartDate = this.get('momentStartDate');
    let momentEndDate = moment.min(this.get('momentEndDate'), moment().endOf('day'));

    this.set('momentEndDate', momentEndDate);

    this.$datepickerStart = this.$().find('input.js-datepicker-start');
    this.$datepickerEnd = this.$().find('input.js-datepicker-end');

    this.$datepickerStart.val(momentStartDate.format(DATE_FORMAT));
    this.$datepickerEnd.val(momentEndDate.format(DATE_FORMAT));

    this._setDatepickerStartByDate();
    this._setDatepickerEndByDate();

    let updateDateObject = {
      startDate: momentStartDate,
      endDate: momentEndDate,
      resolution: DAYS_RESOLUTION
    };

    run.once(this, 'updateDateFilter', updateDateObject);
  },

  willDestroyElement() {
    this._super(...arguments);

    this.$datepickerStart.datepicker('destroy');
    this.$datepickerEnd.datepicker('destroy');
  },

  updateStartAndEndDates: observer('momentStartDate', 'momentEndDate', function () {
    let momentStartDate = this.get('momentStartDate');
    let momentEndDate = this.get('momentEndDate');

    let updateDateObject = {
      startDate: momentStartDate,
      endDate: momentEndDate,
      resolution: DAYS_RESOLUTION
    };

    run.once(this, 'updateDateFilter', updateDateObject);
  }),

  _setDatepickerStartByDate() {
    this.$datepickerStart.datepicker({
      beforeShowDay: null,
      dateFormat: DATE_FORMAT,
      defaultDate: this.get('momentStartDate').toDate(),
      maxDate: this.get('momentEndDate').toDate(),
      nextText: '',
      onClose() {
        run(() => $(this).blur());
      },
      onSelect: (formattedDateString) => {
        run(() => {
          let newMomentStartDate = moment(formattedDateString, DATE_FORMAT);
          this.set('momentStartDate', newMomentStartDate);
        });
      },
      prevText: '',
      selectOtherMonths: true,
      showOtherMonths: true
    });
  },

  _setDatepickerEndByDate() {
    this.$datepickerEnd.datepicker({
      beforeShowDay: null,
      dateFormat: DATE_FORMAT,
      defaultDate: this.get('momentEndDate').toDate(),
      maxDate: moment.max(this.get('momentEndDate'), moment().endOf('day')).toDate(),
      nextText: '',
      onClose() {
        run(() => $(this).blur());
      },
      onSelect: (formattedDateString) => {
        run(() => {
          let newMomentEndDate = moment(formattedDateString, DATE_FORMAT);
          let momentStartDate = this.get('momentStartDate');

          this.set('momentEndDate', newMomentEndDate);

          if (newMomentEndDate.isBefore(momentStartDate)) {
            let newMomentStartDate = moment(newMomentEndDate);

            this.set('momentStartDate', newMomentStartDate);
          }

          this.$datepickerStart.datepicker('option', 'maxDate', newMomentEndDate.toDate());
        });
      },
      prevText: '',
      selectOtherMonths: true,
      showOtherMonths: true
    });
  }
});
