/* globals URI, Twitch, location */

import Component from 'ember-component';
import computed from 'ember-computed';
import ClickOutsideMixin from 'web-client/mixins/click-outside';
import { styledI18n } from 'web-client/helpers/styled-i18n';
import injectService from 'ember-service/inject';
import { normalizeVideoId } from 'web-client/utilities/normalize-video-id';
import {
  facebookShareLink,
  twitterShareLink,
  redditShareLink,
  vkShareLink
} from 'web-client/utilities/social-links';

const PROTOCOL = location.protocol;

export default Component.extend(ClickOutsideMixin, {
  tagName: "span",
  isMenuOpen: false,
  tracking: injectService(),

  _toggleMenu() {
    this.toggleProperty('isMenuOpen');
  },

  clickedOutside() {
    this.set('isMenuOpen', false);
  },

  isChannel: computed.equal('type', 'channel'),
  isVideo: computed.equal('type', 'video'),

  channel: computed("type", "content", "content.channel", function () {
    let type = this.get('type');
    if (type === "channel") {
      return this.get("content");
    } else if (type === "video") {
      return this.get("content.channel");
    }
  }),

  channelUrl: computed('channel.id', function () {
    return Twitch.canonicalUrl.channel(this.get('channel.id'));
  }),

  contentUrl: computed('channelUrl', 'content.url', 'type', function () {
    let type = this.get('type');
    if (type === "channel") {
      return this.get('channelUrl');
    } else if (type === "video") {
      return this.get('content.url');
    }
  }),

  twitterMessage: computed('type', 'content.status', 'content.title', function () {
    let type = this.get('type');
    if (type === "channel") {
      return this.get('content.status');
    } else if (type === "video") {
      return this.get('content.title');
    }
  }),

  embedPlayerCode: computed('type', 'content.id', 'channel.{name,displayName}', 'channelUrl', function () {
    let type = this.get('type');
    let contentId = type === 'video' ? `v${normalizeVideoId(this.get('content.id'))}` : this.get('content.id');
    let disableAutoPlay = type === 'video' ? '&autoplay=false' : '';
    let linkCopy = styledI18n('Watch live video from %{channelDisplayName} on %{hostname}', {
      channelDisplayName: this.get('channel.displayName'),
      hostname: URI().host()
     });

    let embedStyle = 'padding:2px 0px 4px; display:block; width:345px; font-weight:normal; font-size:10px; text-decoration:underline;';

    return `<iframe src="${PROTOCOL}//player.twitch.tv/?${type}=${contentId}${disableAutoPlay}" ` +
           `frameborder="0" allowfullscreen="true" scrolling="no" height="378" width="620"></iframe>` +
           `<a href="${this.get('channelUrl')}?tt_medium=live_embed&tt_content=text_link" ` +
           `style="${embedStyle}">${linkCopy}</a>`;
  }),

  twitterShareLink: computed('type', 'contentUrl', 'content.{status,title}', function () {
    let type = this.get('type');
    let contentUrl = this.get('contentUrl');

    if (type === "channel") {
      let status = this.get('content.status') || '';
      return twitterShareLink(contentUrl, status);
    } else if (type === "video") {
      let title = this.get('content.title') || '';
      return twitterShareLink(contentUrl, title);
    }
  }),

  facebookShareLink: computed('contentUrl', function () {
    return facebookShareLink(this.get('contentUrl'));
  }),

  vkShareLink: computed('contentUrl', function () {
    return vkShareLink(this.get('contentUrl'));
  }),

  redditShareLink: computed('type', 'contentUrl', 'content.{status,title}', function () {
    let type = this.get('type');
    let contentUrl = this.get('contentUrl');

    if (type === "channel") {
      let status = this.get('content.status') || '';
      return redditShareLink(contentUrl, status);
    } else if (type === "video") {
      let title = this.get('content.title') || '';
      return redditShareLink(contentUrl, title);
    }
  }),

  embedChatCode: computed('type', 'content.id', function () {
    let type = this.get('type');
    if (type === "channel") {
      let hostname = URI().host();
      return `<iframe src="${PROTOCOL}//${hostname}/${this.get('content.id')}/chat?popout=" frameborder="0" scrolling="no" height="500" width="350"></iframe>`;
    }
  }),

  trackShare(type) {
      // Git shorthash, for readability.
      let gitShortHash = window.EmberENV.GIT_VERSION && window.EmberENV.GIT_VERSION.slice(0, 7);

      let channel,
          channelId,
          game,
          live,
          location,
          partner,
          vodId,
          vodType;

      if (this.get('type') === 'video') {
        // VOD page, content is the Video Model
        channel = this.get('content.channel.name');
        channelId = this.get('content.owner._id');
        game = this.get('content.game');
        live = false;
        location = 'vod';
        partner = this.get('content.channel.partner');
        vodId = this.get('content.id');
        vodType = this.get('content.broadcast_type');
      } else if (this.get('type') === 'channel') {
        // Live page, content is the Channel Model
        channel = this.get('content.name');
        channelId = this.get('content._id');
        game = this.get('content.game');
        live = true;
        location = 'channel';
        partner = this.get('content.partner');
      }

      let data = {
        app_version: gitShortHash,
        vod_type: vodType,
        vod_id: vodId,
        channel_id: channelId,
        live,
        channel,
        partner,
        game,
        location,
        share_platform: type
      };

      this.get('tracking').trackEvent({
        event: 'video_share',
        data
      });
  },

  actions: {
    toggle() {
      this._toggleMenu();
    },
    trackShare(type) {
      this.trackShare(type);
    },
    selectCopyLink() {
      this.$('.js-copy-link').select();
    }
  }
});
