/* globals Twitch */

import Component from 'ember-component';
import computed, { and, or } from 'ember-computed';
import injectService from 'ember-service/inject';
import ClickOutsideMixin from 'web-client/mixins/click-outside';

export default Component.extend(ClickOutsideMixin, {
  classNames: ['notification-controls', 'cn-notification-controls'],

  storage: injectService(),
  session: injectService(),
  tracking: injectService(),

  init() {
    this._super(...arguments);
    Twitch.sentinel.detect.then((adblock) => {
      this.set('isVisible', !adblock);
      this.trackDisplay(adblock);
    });
  },

  isVisible: false,
  isMenuOpen: false,

  clickedOutside() {
    this.set('isMenuOpen', false);
  },

  trackDisplay(adblock) {
    const event = 'adopt_display';

    if (this.get('canSupport')) {
      this.get('tracking').trackEvent({
        services: ['spade'],
        event: event,
        data: {
          device_id: Twitch.idsForMixpanel.getOrCreateUniqueId(),
          user_id: this.get('session.userData.id'),
          login: this.get('session.userData.login'),
          channel_id: this.get('channel.id'),
          channel: this.get('channel.displayName'),
          adblock: adblock
        }
      });
    }
  },

  trackSupport(support) {
    const event = 'adopt_action';
    const location = 'channel';

    this.get('tracking').trackEvent({
      services: ['spade'],
      event: event,
      data: {
        device_id: Twitch.idsForMixpanel.getOrCreateUniqueId(),
        user_id: this.get('session.userData.id'),
        login: this.get('session.userData.login'),
        channel_id: this.get('channel.id'),
        channel: this.get('channel.displayName'),
        location: location,
        opt_in: support,
        user_agent: window.navigator.userAgent
      }
    });
  },

  userType: computed('session.userData.has_premium', 'session.userData.has_turbo', 'isAdFreeSub', function() {
    const prime = 'Prime user';
    const turbo = 'Turbo user';
    const subscriber = 'subscriber';

    if (this.get('session.userData.has_premium')) {
      return prime;
    }
    if (this.get('session.userData.has_turbo')) {
      return turbo;
    }
    if (this.get('isAdFreeSub')) {
      return subscriber;
    }
  }),

  isChannelOwner: computed('channel.id', function() {
    let { isAuthenticated, userData } = this.get('session');
    return isAuthenticated && userData.login === this.get('channel.id');
  }),

  isAdFreeSub: and('channel.isSubscribed.isSubscribed', 'channel.viewer.has_ad_free_subscription'),

  hasAdFreeExperience: or('isAdFreeSub', 'session.userData.has_turbo'),

  canSupport: and('hasAdFreeExperience', 'channel.partner'),

  isSupporting: computed('channel.id', function() {
    // compare to a string 'true', because local storage always returns as a string
    return this.get(`storage.ad-whitelist-${this.get('channel.id')}`) === 'true';
  }),

  actions: {
    toggleSupport() {
      let key = `ad-whitelist-${this.get('channel.id')}`;
      let value = this.get(`storage.${key}`);

      if (value === 'true') {
        this.set(`storage.${key}`, 'false');
        this.set('isSupporting', false);
        this.set('isMenuOpen', false);
        this.trackSupport(false);
      } else {
        this.set(`storage.${key}`, "true");
        this.set('isSupporting', true);
        this.set('isMenuOpen', true);
        this.trackSupport(true);
      }
    },
    toggleMenu() {
      this.toggleProperty('isMenuOpen');
    }
  }
});
