import EmberObject from 'ember-object';
import Component from 'ember-component';
import computed from 'ember-computed';

/*
  This is a weird component.

  What it attempts to do is manage two tabbed-split-panels.
  One side is the flex side, and one side is the chunk side.

  --------------------------------
  |                 |            |
  |                 |            |
  |                 |            |
  |                 |            |
  |      Flex       |    Chunk   |
  |      1.6gU      |     1gU    |
  |                 |            |
  |                 |            |
  |                 |            |
  |                 |            |
  --------------------------------

  The flex size will vary in size from 1 gridUnit to 2 gridUnit

  As the size of the overall container increases the Flex side will expand to
  consume all of the incremental space, until it grows to the point of adding
  a new complete gridUnit at which point the Chunk side increases by a gridUnit
  and the flex side returns to a single gridUnit in width.

  If the TabbedSplitPanel component is smaller than 2 gridUnits meaning that it
  can't show both the flex and the chunk next to one another, it turns into a
  tabbed view.  Tabs register themselves through the provided action.
*/
export default Component.extend({
  tagName: 'ul',
  classNameBindings: ['isTabbed:tabbed-split--is-tabbed:tabbed-split'],
  gridUnit: 320,
  gutter: 30,
  subGutter: 20,
  tabs: null,

  // These are all scoped an aliased on a subobject so that they can be shared
  // to the panels
  layoutProps: null,
  isTabbed: computed.alias('layoutProps.isTabbed'),
  activeTab: computed.alias('layoutProps.activeTab'),
  flexWidth: computed.alias('layoutProps.flexWidth'),
  chunkWidth: computed.alias('layoutProps.chunkWidth'),
  columns: computed.alias('layoutProps.columns'),

  init() {
    this._super(...arguments);

    this.set('tabs', []);

    this.set('layoutProps', EmberObject.create({
      columns: this.get('columns'),
      isTabbed: this.get('isTabbed'),
      flexWidth: this.get('flexWidth'),
      chunkWidth: this.get('chunkWidth'),
      activeTab: this.get('activeTab'),
      registerTab: (label) => {
        let tabs = this.get('tabs');
        if (tabs.length === 0) {
          this.set('activeTab', label);
        }
        tabs.addObject(label);
      }
    }));
  },

  didUpdateAttrs() {
    this._super(...arguments);
    this.reflow();
  },

  didInsertElement() {
    this._super(...arguments);
    this.reflow();

    this.addEventListener(window, 'resize', () => {
      this.reflow();
    });
  },

  reflow() {
    let width = this.$().width();
    let gridUnit = this.get('gridUnit');
    let gutterSize = this.get('gutter');
    let subGutterSize = this.get('subGutter');

    let columns = Math.floor(width / (gridUnit + gutterSize));
    let chunkWidth = (columns - 1) * gridUnit + Math.max(0, (columns - 2)) * subGutterSize;
    let flexWidth = width - (chunkWidth + gutterSize);
    if (columns < 2) {
      this.set('isTabbed', true);
      this.set('flexWidth', width);
      this.set('chunkWidth', width);
    } else {
      this.set('isTabbed', false);
      this.set('flexWidth', flexWidth);
      this.set('chunkWidth', chunkWidth);
    }
    this.set('columns', columns);
  },

  lockHeight () {
    let height = this.$().height();
    this.$().height(height);
    this.runTask(() => {
      this.$().height('auto');
    });
  },

  actions: {
    activateTab(tab) {
      let tabs = this.get('tabs');
      this.lockHeight();
      if (tabs.includes(tab)) {
        this.set('activeTab', tab);
      }
    }
  }
});
