import Component from 'ember-component';
import computed from 'ember-computed';
import { assert } from 'ember-metal/utils';
import ArrayProxy from 'ember-controller/proxy';
import injectService from 'ember-service/inject';
import DeprecatedStream from 'web-client/models/deprecated-stream';
import DeprecatedVideo from 'web-client/models/deprecated-video';

export const COMMUNITIES = 'COMMUNITIES';
export const CHANNELS = 'CHANNELS';
export const VODS = 'VODS';
const VALID_PERIODS = ['all', 'month', 'week'];
const DEFAULT_PERIOD = 'all';

export default Component.extend({
  api: injectService(),
  search: injectService(),
  store: injectService(),
  tracking: injectService(),

  broadcasterLanguage: null,
  game: null,
  period: DEFAULT_PERIOD,
  showCommunityTab: false,
  showingCommunities: computed.equal('streamList', COMMUNITIES),
  showingChannels: computed.equal('streamList', CHANNELS),
  showingVods: computed.equal('streamList', VODS),
  streamList: CHANNELS,
  tag: null,

  hashtagName: computed('tag.name', function () {
    return this.get('tag.name') || 'all';
  }),

  didReceiveAttrs() {
    this._super(...arguments);
    assert('{{tag-streams}} requires a game attribute', !!this.get('game.name'));
    assert('{{tag-streams}} requires a period attribute', this.get('period') && VALID_PERIODS.indexOf(this.get('period') < 0));
  },

  model: computed('game.id', 'tag.id', 'streamList', 'period', 'broadcasterLanguage', function() {
    let game = this.get('game.name');
    let community = this.get('tag.name');
    let broadcasterLanguage = this.get('broadcasterLanguage');
    if (this.get('showingCommunities')) {
      return this.get('store').query('twitch-community', { game: 'Creative', limit: 100, sortBy: 'viewers' });
    }
    else if (this.get('showingChannels')) {
      if (community) {
        return DeprecatedStream.find('community', { game, community, broadcasterLanguage }).load();
      }
      return DeprecatedStream.find('bygame', { game, broadcaster_language: broadcasterLanguage }).load();
    }
    else if (this.get('showingVods')) {
      let period = this.get('period');
      if (community) {
        return DeprecatedVideo.find('tags', { search: this.get('search'), game, tag: community, period }).load();
      }
      return DeprecatedVideo.find('game', { game, period }).load();
    }
  }),

  trackContentTab() {
    let data = { content: this.get('streamList') };
    this.get('tracking').trackEvent({ event: 'tag_streams_tab', data });
  },

  actions: {
    loadMore() {
      this.get('model').loadMore(true).then(response => {
        if (response.hideMore) {
          let data = { community: this.get('tag.name') };
          this.get('tracking').trackEvent({ event: 'community_streams_bottom', data });
        }
      }, e => {
        if (
          e.constructor === DeprecatedStream ||
          e.constructor === DeprecatedVideo ||
          ArrayProxy.detectInstance(e)
        ) {
          /* The deprecated stream and video models throw an error with the model as
           * an error when a 404 occurs. In these cases, return a null model */
          return;
        }
        throw e;
      });
    },

    showCommunities() {
      this.set('streamList', COMMUNITIES);
      this.trackContentTab();
    },

    showChannels() {
      this.set('streamList', CHANNELS);
      this.trackContentTab();
    },

    showVods() {
      this.set('streamList', VODS);
      this.trackContentTab();
    }
  }
});
