import Component from 'ember-component';
import computed from 'ember-computed';
import injectService from 'ember-service/inject';
import { htmlSafe } from 'ember-string';
import { A as emberA } from 'ember-array/utils';
import Ember from 'ember';
import { checkActionMessage } from 'web-client/helpers/twitch-conversations/conversation-line-helpers';
import tmiEmotes from 'web-client/utilities/tmi-emotes';
import {
  linkifyMessage,
  mentionizeMessage,
  emoticonizeMessage,
  finalizeTokens,
  capitalize
} from 'web-client/helpers/chat/chat-line-helpers';
import { formatDisplayName } from 'web-client/helpers/format-display-name';


// TODO find a way to import this
const { Handlebars: { Utils: { escapeExpression } } } = Ember;

/* Input:
  Message Model

  Output:
  <div class='conversation-chat-line outgoing|incoming>
    <span class='from' style='color:#fff'>Extalix</span>
    <span class='colon'>:</span>
    <span class='message' style='{{if actionMessage}}color:#fff{{/if}}'>message text here</span>
  </div>
*/

export default Component.extend({
  classNameBindings: [':conversation-chat-line', 'isActionMessage:action', 'fromSelf:outgoing:incoming'],
  // nonce is used to track whisper reliability. This is used by web-client/services/whispers-shim
  // as of 4/12/16
  attributeBindings: ['message.sentAt:title', 'message.nonce:data-nonce'],
  session: injectService(),
  currentLogin: computed.readOnly('session.userData.login'),
  currentUserId: computed.readOnly('session.userData.id'),

  fromSelf: computed('message.from.id', 'currentUserId', function () {
    return this.get('message.from.id') === this.get('currentUserId').toString();
  }),

  fromDisplay: computed('message.from.displayName', 'message.from.username', function () {
    if (this.get('message.from.displayName')) {
      let displayName = escapeExpression(this.get('message.from.displayName'));
      let username = escapeExpression(this.get('message.from.username'));
      return formatDisplayName(username, displayName);
    }
    return capitalize(this.get('message.from.username'));
  }),

  isActionMessage: computed('message.body', function () {
    return this.get('message.body').substr(0, 4) === '/me ';
  }),

  emoticons: computed('message.tags.emotes', function () {
    let emotes = this.get('message.tags.emotes');
    if (emotes) {
      return tmiEmotes(emotes);
    }
    return [];
  }),

  tokenizedMessage: computed('message.body', 'emoticons.[]', 'currentLogin', function () {
    let tokens = [this.get('message.body')],
        emoticons = this.get('emoticons');
    let spamLikelihood = this.get('message.thread.spamLikelihood');

    tokens = checkActionMessage(tokens);
    if (spamLikelihood === 'low') {
      tokens = linkifyMessage(tokens, false);
    }
    tokens = mentionizeMessage(tokens, this.get('currentLogin'), this.get('fromSelf'));
    tokens = emoticonizeMessage(tokens, emoticons);
    return emberA(finalizeTokens(tokens));
  }),

  fromStyle: computed('message.from.color', function () {
    let style = '';
    let color = this.get('message.from.color');
    if (color) {
      style = `color: ${color};`;
    }
    return htmlSafe(style);
  }),

  messageStyle: computed('message.from.color', 'isActionMessage', function () {
    let style = '';
    if (this.get('isActionMessage')) {
      let color = this.get('message.from.color');
      if (color) {
        style = `color: ${color};`;
      }
    }
    return htmlSafe(style);
  })
});
