/* globals i18n */
import Component from 'ember-component';
import computed from 'ember-computed';
import { formatDisplayName } from 'web-client/helpers/format-display-name';
import { A as emberA } from 'ember-array/utils';
import injectService from 'ember-service/inject';
import observer from 'ember-metal/observer';
import { libraries as libs } from 'web-client/utilities/conversations-util';
import {
  emoticonizeMessage,
  finalizeTokens
} from 'web-client/helpers/chat/chat-line-helpers';
import tmiEmotes from 'web-client/utilities/tmi-emotes';
import { checkActionMessage } from 'web-client/helpers/twitch-conversations/conversation-line-helpers';

export default Component.extend({
  api: injectService(),
  conversations: injectService('twitch-conversations/conversations'),
  session: injectService(),

  avatarUrl: libs.defaultAvatar,
  shouldShowInput: false,

  allUserEmotes: computed.alias('conversations.allUserEmotes'),
  message: computed.alias('thread.mostRecentMessage'),

  unreadString: computed('thread.unreadCount', 'otherDisplayName', function() {
    let count = this.get('thread.unreadCount');
    return `sent you ${count} new whisper${count === 1 ? '' : 's'}`;
  }),

  otherDisplayName: computed('otherUser', function () {
    return formatDisplayName(this.get('otherUser.username'), this.get('otherUser.displayName'));
  }),

  replyPlaceholder: computed('otherDisplayName', function() {
    let translatedString = i18n('Reply directly to @{{name}}');
    return translatedString.replace('{{name}}', this.get('otherDisplayName'));
  }),

  emoticons: computed('message.tags.emotes', function () {
    let emotes = this.get('message.tags.emotes');
    if (emotes) {
      return tmiEmotes(emotes);
    }
    return [];
  }),

  tokenizedMessage: computed('message.body', 'emoticons.[]', function() {
    let tokens = [this.get('message.body')];
    let emoticons = this.get('emoticons');

    tokens = checkActionMessage(tokens);
    tokens = emoticonizeMessage(tokens, emoticons);
    return emberA(finalizeTokens(tokens));
  }),

  onRead: observer('thread.unreadCount', function() {
    if (!this.get('thread.unreadCount')) {
      this.sendAction('close', true);
    }
  }),

  focusIn() {
    this.set('isFocused', true);
    this.sendAction('cancelTimer');
  },

  focusOut() {
    this.set('isFocused', false);
    this.sendAction('startTimer');
  },

  mouseEnter() {
    this.sendAction('cancelTimer');
  },

  mouseLeave() {
    if (!this.get('isFocused')) {
      this.sendAction('startTimer');
    }
  },

  init() {
    this._super(...arguments);
    let otherUser = this.get('thread.participants').rejectBy('id', this.get('session.userData.id').toString()).objectAt(0);
    this.set('otherUser', otherUser);

    this.get('api').request('get', `users/${otherUser.get('username')}`).then(otherUserData => {
      if (!this.isDestroyed && otherUserData.logo) {
        this.set('avatarUrl', otherUserData.logo);
      }
    });
  },

  actions: {
    muteConversation() {
      this.get('thread').set('isMuted', true);
      this.get('thread').save();
      this.sendAction('close');
    },

    showInput() {
      this.set('shouldShowInput', true);
      this.runTask(() => this.$('.qr-input__container textarea').focus());
    },

    onBodyClick() {
      let login = this.get('otherUser.username');
      this.get('conversations').openConversation(login);
      this.sendAction('close');
    }
  }

});
