/* globals Twitch, i18n */

import Component from 'ember-component';
import computed from 'ember-computed';
import Ember from 'ember';
import injectService from 'ember-service/inject';
import { htmlSafe } from 'ember-string';
import { getRichPresence } from 'web-client/utilities/presence-util';
import $ from 'jquery';

const { Handlebars } = Ember;
const { escapeExpression } = Handlebars.Utils;

const ProfileCardComponent = Component.extend({
  store: injectService(),
  tracking: injectService(),
  conversations: injectService('twitch-conversations/conversations'),
  session: injectService(),
  presence: injectService('twitch-presence/presence'),
  friends: injectService('twitch-friends-list/friends'),

  friendInfo: null,
  profileId: null,
  profile: null,
  showActions: false,
  request: false,
  isMe: null,

  showFollow: true,
  showWhisper: true,

  classNames: ['profile-card'],
  classNameBindings: ['showActions:profile-card--show-actions'],

  didReceiveAttrs() {
    this._super(...arguments);

    let store = this.get('store');
    let id = this.get('profileId');
    let me = Twitch.user.login();

    if (id === this.get('profile.id')) { return; }

    store.findRecord('profile', id).then(profile => {
      if (this.isDestroyed) { return; }

      this.set('profile', profile);
      this.set('isMe', me === this.get('profile.login'));

      return store.findRecord('channel', profile.get('login'));
    }).then(channel => {
      if (this.isDestroyed) { return; }

      this.set('channelInfo', channel);
      return store.peekRecord('friends-list-user', id);
    }).then(friend => {
      if (this.isDestroyed) { return; }
      if (friend && friend.get('isFriend')) {
        this.set('friendInfo', friend);
      }
    });
  },

  mouseEnter() {
    this.set('canShowMenu', true);
  },

  mouseLeave() {
    this.set('canShowMenu', false);
  },

  bannerStyle: computed('profile.{profile_banner_background_color,profile_banner}', function () {
    let color = this.get('profile.profile_banner_background_color');
    let banner = this.get('profile.profile_banner');
    let backgroundColorStyle = color ? `background-color: ${ escapeExpression(color) };` : '';
    let backgroundImageStyle = banner ? `background-image: url(${ escapeExpression(banner) });` : '';

    return htmlSafe(`${ backgroundColorStyle } ${ backgroundImageStyle }`);
  }),

  channelUrl: computed('channelInfo.id', function () {
    let channelId = this.get('channelInfo.id');

    return Twitch.canonicalUrl.channel(channelId);
  }),

  statusString: computed('friendInfo', function() {
    let friend = this.get('friendInfo');
    return getRichPresence(friend.get('availability'), friend.get('activities')[0]);
  }),

  location: computed('friendInfo', function() {
    let activity = this.get('friendInfo').get('activities')[0];
    if (activity.type === 'none') {
      if (this.get('activity').channel_login) {
        return `/${activity.channel_login}?ref=click_profile_card_presence`;
      } else if (['broadcasting', 'streaming'].includes(activity.type)) {
        return `/${this.get('user.login')}?ref=click_profile_card_presence`;
      }
    }
  }),

  bioString: computed('profile.bio', function() {
    let bio = this.get('profile.bio') || 'This user has no bio.';
    return i18n(bio);
  }),

  actions: {
    message() {
      if (!this.get('session.isAuthenticated')) {
        return $.showAuthenticationOverlay('login', {
          mpSourceAction: 'whisper'
        });
      }

      let profileLogin = this.get('profile.login');
      this.get('conversations').openConversation(profileLogin);
    }
  }
});

ProfileCardComponent.reopenClass({
  positionalParams: ['profileId']
});

export default ProfileCardComponent;
