import Component from 'ember-component';
import { libraries as lib } from 'web-client/utilities/vod-appeals-util';
import { getReasonDescription } from 'web-client/components/video/appeals/reason-selector/component';
import computed from 'ember-computed';
import { assert } from 'ember-metal/utils';

/*
  Muted Tracks and Track Appeals

  Required:
    * track: track returned from muted_tracks endpoint
    One of the following must be true: (TODO: remove/refactor out this requirement)
      * selectTracks: all tracks, with options to select reasons for appealable tracks
      * selectedTracks: selected tracks only (before appeal submission)
      * pendingTracks: tracks related to a pending vod appeal
*/
export default Component.extend({
  tagName: 'tr',
  classNames: [ 'music-item' ],
  classNameBindings: [ 'canAppealTrack::disabled' ],

  isSelectingReason: false,
  isMuted: computed.alias('track.is_muted'),
  offsetInSeconds: computed.alias('track.offset'),
  reasonForAppealing: computed.alias('track.reasonForAppealing'),

  // context, chooses template and tracks to render (passed into component)
  // TODO: refactor this polymorphic, complex component.
  selectTracks: false, // show all tracks with actions
  selectedTracks: false, // show tracks selected for appealing
  pendingTracks: false, // show tracks with an appeal in progress

  init() {
    this._super(...arguments);

    assert(
      `${this} requires one of selectedTracks, selectTracks, or pendingTracks as an attr`,
      this.get('selectTracks') || this.get('selectedTracks') || this.get('pendingTracks')
    );
  },

  // Used for rendering only a subset of muted tracks depending on the context supplied
  shouldRenderTrack: computed(
    'selectTracks', 'pendingTracks', 'selectedTracks',
    'reasonForAppealing', 'track.track_appeal.status',
  function() {
    if (this.get('selectTracks')) {
      return true;
    } else if (this.get('selectedTracks')) {
      return !!this.get('reasonForAppealing');
    } else if (this.get('pendingTracks')) {
      return !!this.get('reasonForAppealing') || this.get('track.track_appeal.status') === "pending";
    }
  }),

  canAppealTrack: computed('track.{track_appeal,is_muted}', function() {
    let appealExists = !!this.get('track.track_appeal');
    let isMuted = this.get('track.is_muted');

    return !appealExists && isMuted;
  }),

  offsetFormatted: computed('offsetInSeconds', function() {
    return lib.display.lengthAsClock(this.get('offsetInSeconds'));
  }),

  // If an appeal already exists, show the reason it was appealed on
  // Else show what reason the user is going to appeal this track with.
  selectedReason: computed('track.reasonForAppealing', 'track.track_appeal.reason', function() {
    let appealReason = this.get('track.track_appeal.reason');
    let reasonForAppealing = this.get('track.reasonForAppealing');

    if (appealReason) {
      return getReasonDescription(appealReason);
    } else if (reasonForAppealing) {
      return reasonForAppealing.description;
    }

    return 'Select Reason';
  }),

  actions: {
    selectReason(reason) {
      this.set('isSelectingReason', false);
      if (reason.reason !== 'none') {
        this.set('track.reasonForAppealing', reason);
      } else {
        this.set('track.reasonForAppealing', '');
      }
    },
    toggleIsSelectingReason() {
      this.toggleProperty('isSelectingReason');
    },
    stopSelectingReason() {
      this.set('isSelectingReason', false);
    },
    clearReasonSelector() {
      this.set('track.reasonForAppealing', '');
    }
  }
});
