/* global i18n */

import Controller from 'ember-controller';
import injectService from 'ember-service/inject';
import computed from 'ember-computed';
import { DISCLAIMER_COUNTRIES_MAPPING } from 'web-client/utilities/bits/disclaimer-countries-mapping';

const TIMS_SUCCESS = 'success';
const TIMS_NO_ACTION = 'no_action';
const SAVE_SUCCESS = i18n('Bits Agreement Changes Are Saved');
const SAVE_ERROR = i18n('Bits Agreement Changes Failed');

export default Controller.extend({
  bits: injectService(),
  globals: injectService(),
  notify: injectService(),
  geo: injectService(),

  isSaving: false,
  isBitsAmendmentModalOpen: false,
  isBitsAmendmentConfirmationModalOpen: false,
  minBits: computed.alias('model.minBits'),
  minBitsEmote: computed.alias('model.minBitsEmote'),
  bitsBadges: computed.alias('model.bitsBadges'),
  bitsBadgeTiers: computed.alias('model.bitsBadgeTiers'),
  user: computed.alias('model.user'),
  amendmentSigned: computed.alias('model.amendmentSigned'),
  bitsOnboarded: computed.alias('model.bitsOnboarded'),
  pinRecentCheers: computed.alias('model.pinRecentCheers'),
  pinTopCheers: computed.alias('model.pinTopCheers'),
  recentCheerMin: computed.alias('model.recentCheerMin'),
  recentCheerTimeout: computed.alias('model.recentCheerTimeout'),
  pinCheerEnabled: computed.readOnly('globals.pCheerEnabled'),
  userGeo: null,

  enableTimeout: computed.alias('model.enableTimeout'),

  requiresRoyaltyTaxAction: computed('model.product.royaltyTaxState', function () {
    let royaltyTaxState = this.get('model.product.royaltyTaxState');

    return royaltyTaxState !== TIMS_SUCCESS && royaltyTaxState !== TIMS_NO_ACTION;
  }),

  showAmendmentDisclaimer: computed('userGeo', function() {
    if (DISCLAIMER_COUNTRIES_MAPPING.indexOf(this.get('userGeo')) !== -1) {
      return true;
    }
  }),

  requiresServiceTaxAction: computed.alias('model.requiresServicesTax'),

  pendingBitsAmendmentPreRequisites: computed.or('requiresRoyaltyTaxAction', 'requiresServiceTaxAction'),
  isBitsAmendmentAccepted: computed.and('amendmentSigned', 'bitsOnboarded'),

  init() {
    this._super(...arguments);

    this.get('geo').getGeo().then(response => {
      if (this.isDestroyed) { return; }
      this.set('userGeo', response.geo);
    });
  },

  _setBitsAmendment(signed) {
    this.set('amendmentSigned', signed);
    this.set('bitsOnboarded', signed);
  },

  _saveBitsAmendment() {
    if (this.get('isSaving')) { return; }

    let notify = this.get('notify');

    this.set('isSaving', true);

    return this.get('bits').setOnboarded(this.get('user._id')).then(() => {
      if (this.isDestroyed) { return; }

      notify.success(SAVE_SUCCESS);
    }, () => {
      if (this.isDestroyed) { return; }

      this._setBitsAmendment(false);
      notify.error(SAVE_ERROR);
    }).finally(() => {
      if (this.isDestroyed) { return; }

      this.set('isSaving', false);
    });
  },

  isBitsAmendmentAcceptedAndSavedOrSaving: computed.or('isBitsAmendmentAccepted', 'isSaving'),
  isBitsAmendmentDisabled: computed.or('isBitsAmendmentAcceptedAndSavedOrSaving', 'pendingBitsAmendmentPreRequisites'),

  actions: {
    syncAndOpenBitsAmendmentModal(isAccepted) {
      if (this.get('isBitsAmendmentModalOpen') || this.get('isBitsAmendmentConfirmationModalOpen')) { return; }

      this._setBitsAmendment(isAccepted);

      if (isAccepted) {
         this.set('isBitsAmendmentModalOpen', true);
      }
    },

    // Actions for Bits Amendment modal
    closeBitsAmendmentModal(signed) {
      if (!this.get('isBitsAmendmentModalOpen') || this.get('isBitsAmendmentConfirmationModalOpen')) { return; }

      this.set('isBitsAmendmentModalOpen', false);

      if (!signed) {
        this._setBitsAmendment(false);
        return;
      }

      this.set('isBitsAmendmentConfirmationModalOpen', true);
    },

    closeBitsAmendmentConfirmationModal(confirmed) {
      this._setBitsAmendment(confirmed);
      if(confirmed) {
        this._saveBitsAmendment();
      }

      this.set('isBitsAmendmentConfirmationModalOpen', false);
    }
  }
});
