/* global i18n */

import Controller from 'ember-controller';
import injectService from 'ember-service/inject';
import computed from 'ember-computed';
import EmberObject from 'ember-object';

const HASHTAG_REQUIREMENT = i18n("Hashtags cannot contain '#' '@' or whitespace.");
const MATCH_ERROR = i18n('There was a problem ending the match. Please try again.');
const MATCH_ENDED = i18n('The match has been successfully ended.');
const SAVE_ERROR = i18n('There was a problem updating hashtags. Please try again.');
const SAVE_SUCCESS = i18n('Your hashtags have been successfully saved');
const TWO_TAG_REQUIREMENT = i18n('Exactly two hashtags must be selected.');

export default Controller.extend({
  bits: injectService(),
  notify: injectService(),

  isSaving: false,

  tags: computed.map('hashtags', function (tag) {
    return EmberObject.create({ hashtag: tag.hashtag, selected: tag.selected });
  }),

  actions: {
    save() {
      if (this.get('isSaving')) { return; }

      let notify = this.get('notify');
      let tags = this.get('tags');
      let hashtags = [];
      let count = 0;

      for(let i = 0; i<tags.length; i++) {
        if (tags[i].selected === true) {
          count++;
        }

        hashtags.push({ hashtag: tags[i].hashtag, selected: tags[i].selected });
      }

      if (count !== 2) {
        return notify.error(TWO_TAG_REQUIREMENT);
      }

      this.set('isSaving', true);

      this.get('bits').setHashtags(this.get('user._id'), { hashtags: hashtags }).then(() => {
        if (this.isDestroyed) { return; }

        notify.success(SAVE_SUCCESS);
      }, () => {
        if (this.isDestroyed) { return; }

        notify.error(SAVE_ERROR);
      }).finally(() => {
        if (this.isDestroyed) { return; }

        this.set('isSaving', false);
      });
    },

    addTag() {
      let newTag = this.get('newTag');
      let notify = this.get('notify');
      let tags = this.get('tags');
      let newHashtags = [];

      if (newTag.match(/#|@|\s/)) {
        return notify.error(HASHTAG_REQUIREMENT);
      }

      for (let i = 0; i < tags.length; i++) {
        newHashtags.push({ hashtag: tags[i].hashtag, selected: tags[i].selected });
      }

      newHashtags.push({ hashtag: newTag, selected: false });

      this.set('hashtags', newHashtags);
      this.set('newTag', '');
    },

    removeTag(tag) {
      let tags = this.get('tags');
      let newHashtags = [];

      for (let i = 0; i < tags.length; i++) {
        if (tags[i] !== tag) {
          newHashtags.push({ hashtag: tags[i].hashtag, selected: tags[i].selected });
        }
      }

      this.set('hashtags', newHashtags);
    },

    endMatch() {
      let notify = this.get('notify');
      let tags = this.get('tags');
      let channelID = this.get('user._id');
      let hashtags = tags.map(tag => {
        return { hashtag: tag.hashtag, selected: false };
      });

      this.set('hashtags', hashtags);
      this.set('isSaving', true);

      this.get('bits').setHashtags(channelID, { hashtags }).then(() => {
        if (this.isDestroyed) { return; }

        notify.success(MATCH_ENDED);
      }, () => {
        if (this.isDestroyed) { return; }

        notify.error(MATCH_ERROR);
      }).finally(() => {
        if (this.isDestroyed) { return; }

        this.set('isSaving', false);
      });
    }
  }
});
