/* global i18n */

import Controller from 'ember-controller';
import injectService from 'ember-service/inject';
import computed from 'ember-computed';

const TIMS_SUCCESS = 'success';
const TIMS_NO_ACTION = 'no_action';
const SAVE_SUCCESS = i18n('Game Commerce Agreement Changes Are Saved');
const SAVE_ERROR = i18n('Game Commerce Bounty Agreement Changes Failed');

export default Controller.extend({
  bounties: injectService(),
  globals: injectService(),
  notify: injectService(),
  tracking: injectService(),

  isSaving: false,
  isAmendmentModalOpen: false,
  isAmendmentConfirmationModalOpen: false,
  user: computed.alias('model.user'),
  amendmentAccepted: computed.alias('model.amendmentAccepted'),

  requiresRoyaltyTaxAction: computed('model.product.royaltyTaxState', function () {
    let royaltyTaxState = this.get('model.product.royaltyTaxState');
    return royaltyTaxState !== TIMS_SUCCESS && royaltyTaxState !== TIMS_NO_ACTION;
  }),

  requiresServiceTaxAction: computed.alias('model.requiresServicesTax'),

  pendingAmendmentPreRequisites: computed.or('requiresRoyaltyTaxAction', 'requiresServiceTaxAction'),

  _setAmendment(accepted) {
    this.set('amendmentAccepted', accepted);
  },

  _saveAmendment() {
    if (this.get('isSaving')) { return; }

    let notify = this.get('notify');

    this.set('isSaving', true);

    return this.get('bounties').setGameCommerceAmendmentAcceptance(this.get('user._id')).then(() => {
      if (this.isDestroyed) { return; }

      let eventData = {
        channel: this.get('user.display_name'),
        channel_id: this.get('user._id').toString(),
        language: this.get('user.broadcaster_language'),
        completed: true
      };

      this.get('tracking').trackEvent({
        event: 'fuel_agreement_form',
        data: eventData
      });

      notify.success(SAVE_SUCCESS);
    }, () => {
      if (this.isDestroyed) { return; }

      this._setAmendment(false);
      notify.error(SAVE_ERROR);
    }).finally(() => {
      if (this.isDestroyed) { return; }

      this.set('isSaving', false);
    });
  },

  isAmendmentDisabled: computed.or('amendmentAccepted', 'pendingAmendmentPreRequisites'),

  actions: {
    syncAndOpenAmendmentModal(isAccepted) {
      if (this.get('isAmendmentModalOpen') || this.get('isAmendmentConfirmationModalOpen')) { return; }

      this._setAmendment(isAccepted);

      if (isAccepted) {
        this.set('isAmendmentModalOpen', true);
      }
    },

    // Actions for Amendment modal
    closeAmendmentModal(accepted) {
      if (!this.get('isAmendmentModalOpen') || this.get('isAmendmentConfirmationModalOpen')) { return; }

      this.set('isAmendmentModalOpen', false);

      if (!accepted) {
        this._setAmendment(false);

        let eventData = {
          channel: this.get('user.display_name'),
          channel_id: this.get('user._id').toString(),
          language: this.get('user.broadcaster_language'),
          completed: false
        };

        this.get('tracking').trackEvent({
          event: 'fuel_agreement_form',
          data: eventData
        });

        return;
      }

      this.set('isAmendmentConfirmationModalOpen', true);
    },

    closeAmendmentConfirmationModal(confirmed) {
      this._setAmendment(confirmed);
      if(confirmed) {
        this._saveAmendment();
      } else {
        let eventData = {
          channel: this.get('user.display_name'),
          channel_id: this.get('user._id').toString(),
          language: this.get('user.broadcaster_language'),
          completed: false
        };

        this.get('tracking').trackEvent({
          event: 'fuel_agreement_form',
          data: eventData
        });
      }

      this.set('isAmendmentConfirmationModalOpen', false);
    }
  }
});
