/* global i18n */

import Controller from 'ember-controller';
import injectService from 'ember-service/inject';
import computed from 'ember-computed';
import { isPresent } from 'ember-utils';

const SAVE_SUCCESS = i18n('Payment Information Changes Are Saved');
const SAVE_FAILURE = i18n('Payment Information Changes Failed');

export default Controller.extend({
  globals: injectService(),
  notify: injectService(),

  isNetPaymentAmendmentSaving: false,

  netPaymentAmendment: computed.alias('model.netPaymentAmendment'),

  isNetPaymentAmendmentAcceptedByDefault: computed.equal('netPaymentAmendment', 'accepted_by_default'),
  isNetPaymentAmendmentSigned: computed.equal('netPaymentAmendment', 'signed'),
  isNetPaymentAmendmentAccepted: computed.or('isNetPaymentAmendmentAcceptedByDefault', 'isNetPaymentAmendmentSigned'),

  setNetPaymentAmendment(signed) {
    this.set('netPaymentAmendment', signed ? 'signed' : 'unsigned');
  },

  saveNetPaymentAmendment() {
    if (this.get('isNetPaymentAmendmentSaving')) { return; }

    let notify = this.get('notify');

    this.set('isNetPaymentAmendmentSaving', true);

    return this.get('model').save().then(() => {
      if (this.isDestroyed) { return; }

      notify.success(SAVE_SUCCESS);
    }, () => {
      if (this.isDestroyed) { return; }

      notify.error(SAVE_FAILURE);
    }).finally(() => {
      if (this.isDestroyed) { return; }

      this.set('isNetPaymentAmendmentSaving', false);
    });
  },

  hasNoUnsavedChanges: computed.not('model.hasDirtyAttributes'),
  hasNoUnsavedNetPaymentAmendmentChanges: computed('hasNoUnsavedChanges', function() {
    if (this.get('hasNoUnsavedChanges')) {
      return true;
    }
    return !isPresent(this.get('model').changedAttributes().netPaymentAmendment);
  }),
  hasNoUnsavedBitsAmendmentChanges: computed('hasNoUnsavedChanges', function() {
    if (this.get('hasNoUnsavedChanges')) {
      return true;
    }
    return !isPresent(this.get('model').changedAttributes().bitsAmendment);
  }),

  isNetPaymentAmendmentAcceptedAndSaved: computed.and('isNetPaymentAmendmentAccepted', 'hasNoUnsavedNetPaymentAmendmentChanges'),
  isNetPaymentAmendmentAcceptedAndSavedOrSaving: computed.or('isNetPaymentAmendmentAccepted', 'isNetPaymentAmendmentSaving'),
  isNetPaymentAmendmentDisabled: computed.alias('isNetPaymentAmendmentAcceptedAndSavedOrSaving'),

  /*
    Component-powered modals would let us handle this more elegantly.
    Since we don't have that, we instead provide this controller to the modal,
    giving us access to the amendment-related actions below.

    We still have to use the netPaymentAmendment View for the
    scroll-checking behavior it provides us.
  */
  openNetPaymentAmendmentModal() {
    this.send('openInModal', 'netPaymentAmendment', this);
  },

  actions: {
    syncAndOpenNetPaymentAmendmentModal(isAccepted) {
      /*
        The UX here should not be using a checkbox. It should be a button.

        However, while it is, we need to make sure the checkbox stays
        in sync with the underlying property, so that the view rerenders
        as you'd expect.

        Removing value='target.checked' and calling preventDefault()
        doesn't appear to stop the browser from modifying the DOM element
        (therefore letting things stay in sync), so we manually sync instead,
        and uncheck it later.
      */
      this.setNetPaymentAmendment(isAccepted);

      if (isAccepted) {
        this.openNetPaymentAmendmentModal();
      }
    },
    // Actions for Net Payment Amendment modal
    closeNetPaymentAmendmentModal(signed) {
      this.setNetPaymentAmendment(signed);
      this.send('closeModal');

      if (signed) {
        this.saveNetPaymentAmendment();
      }
    }
  }
});
