/* globals Twitch */

import $ from 'jquery';
import Model from 'ember-data/model';
import attr from 'ember-data/attr';
import { hasMany, belongsTo } from 'ember-data/relationships';
import ChannelViewerModel from 'web-client/models/channel-viewer';
import PanelModel from 'web-client/models/panel';
import RoomModel from 'web-client/models/room';
import StreamModel from 'web-client/models/stream';
import TicketProductModel from 'web-client/models/ticket-product';
import DeprecatedVideo from 'web-client/models/deprecated-video';
import computed from 'ember-computed';
import injectService from 'ember-service/inject';
import { JTV_USER_PICTURES_404_USER_70X70_URL } from 'web-client/utilities/urls/static-cdn';
import config from 'web-client/config/environment';
import { htmlSafe } from 'ember-string';
import { ALL_VIDEOS_PARAM, ARCHIVE_TYPE_PARAM, UPLOAD_TYPE_PARAM } from 'web-client/utilities/video/types';

export default Model.extend({
  globals: injectService(),

  /* Attributes Start */
  _id: attr('number'),
  broadcasterLanguage: attr('string'),
  creative: attr(),
  channelFeedEnabled: attr('boolean'),
  community: attr('string'),
  delay: attr('number'),
  delayControlsEnabled: attr('boolean'),
  displayName: attr('string'),
  game: attr('string'),
  lastAccessedAt: attr('date'),
  logo: attr('string', { defaultValue: JTV_USER_PICTURES_404_USER_70X70_URL }),
  mature: attr('boolean'),
  name: attr('string'),
  partner: attr(),
  primaryTeamDisplayName: attr('string'),
  primaryTeamName: attr('string'),
  profileBanner: attr('string'),
  profileBannerBackgroundColor: attr('string'),
  showChat: attr('boolean'),
  status: attr('string'),
  teespringCampaignEnabled: attr('boolean'),
  videoBanner: attr('string'),
  views: attr('number'),
  /* Attributes End */

  /* Relationships Start */
  clips: hasMany('clip', { async: true }),
  events: hasMany('event'),
  followers: hasMany('user'),
  following: hasMany('channel', { inverse: null }),
  directoryFollowing: hasMany('channel', { inverse: null }),
  sidebarFollowing: hasMany('channel', { inverse: null }),
  collections: hasMany('video-playlist', { async: false }),
  user: belongsTo('user'),
  stream: belongsTo('stream'),
  /* Relationships End */

  hostModeTarget: null,

  installedExtensions: [],

  fetchInstalledExtensions() {
    return this.get('store').query('extension-installation', {
      channelId: this.get('_id')
    }).then((data) => {
      this.set('installedExtensions', data);
    });
  },

  channelUrl: computed('id', function () {
    return Twitch.uri.channel(this.get('id'));
  }),

  panels: computed(function () {
    return PanelModel.find('user', {user: this.get('id')});
  }),

  videos: computed(function() {
    return DeprecatedVideo.find('channel', {channel: this.get('id'), broadcastType: ALL_VIDEOS_PARAM });
  }),

  pastBroadcasts: computed(function () {
    return DeprecatedVideo.find('channel', {channel: this.get('id'), broadcastType: ARCHIVE_TYPE_PARAM});
  }),

  uploads: computed(function() {
    return DeprecatedVideo.find('channel', {channel: this.get('id'), broadcastType: UPLOAD_TYPE_PARAM });
  }),

  highlights: computed(function () {
    return DeprecatedVideo.find('channel', {channel: this.get('id')});
  }),

  product: computed(function () {
    return TicketProductModel.findOne(this.get('id'));
  }),

  room: computed(function () {
    return RoomModel.findOne(this.get('id'));
  }),

  relatedChannels: computed(function () {
    return StreamModel.find('related', {channel: this.get('id')});
  }),

  viewer: computed('name', function() {
    let viewer = ChannelViewerModel.findOne(this.get('name'));
    viewer.load();
    return viewer;
  }),

  followingTotal: computed.readOnly('following.content.meta.total'),
  followersTotal: computed.readOnly('followers.content.meta.total'),

  // TODO remove, defensive alias until we remove all 'display_name'
  display_name: computed('displayName', function() {
    if (config.environment !== 'production') {
      console.warn('all usage of channel.display_name should be changed to channel.displayName');
    }

    return this.get('displayName');
  }),

  // Enables Broadcaster Language Chat, which restricts chat to viewers of the same language as the broadcast
  isLanguageRestricted: computed('broadcasterLanguage', function() {
    let langCode = this.get('broadcasterLanguage');
    let defaultBLCLanguages = this.get('globals.defaultBLCLanguages');
    return $.inArray(langCode, defaultBLCLanguages) >= 0;
  }),

  isLanguageOther: computed('broadcasterLanguage', function() {
    let isOther = this.get('broadcasterLanguage') === 'other';
    return htmlSafe(isOther.toString());
  })
});
