/* globals Twitch */
import Model from 'ember-data/model';
import attr from 'ember-data/attr';
import { belongsTo } from 'ember-data/relationships';
import computed from 'ember-computed';
import { assert } from 'ember-metal/utils';
import { UPLOAD_TYPE_PARAM } from 'web-client/utilities/video/types';
import ChannelViewerModel from 'web-client/models/channel-viewer';

const PREVIEW_SIZES = ['small', 'medium', 'large'];
const DEFAULT_STATUS = 'new';

const PUBLIC = 'public';

export default Model.extend({
  channel: belongsTo('channel', { async: true }),
  channelName: attr('string'),
  createdAt: attr('string'),
  deleteCustomThumbnail: attr('boolean', { defaultValue: false }),
  description: attr('string'),
  descriptionMarkdown: attr('string'),
  descriptionHTML: attr('string'),
  game: attr('string'),
  isMuted: attr('boolean'),
  length: attr('number'),
  // An object of thumbnails at different sizes to use as a preview
  preview: attr(),
  publishedAt: attr('string'),
  recordedAt: attr('string'),
  status: attr('string', { defaultValue: DEFAULT_STATUS }),
  tagList: attr('string'),
  thumbnails: attr(),
  thumbnailIndex: attr('number'),
  title: attr('string'),
  // maps to broadcast_type
  type: attr('string'),
  url: attr('string'),
  viewable: attr('string'),
  views: attr('number'),

  customThumbnail: computed('thumbnails', 'thumbnailIndex', function () {
    return this.get('thumbnails').findBy('type', 'custom');
  }),

  gameBoxArt: computed('game', function() {
    let game = this.get('game');
    return Twitch.uri.gameBoxArtJpg(game);
  }),

  gameUrl: computed('game', function() {
    let game = this.get('game');
    return Twitch.uri.gameVideos(game);
  }),

  isPublished: computed.equal('viewable', PUBLIC),

  recordedDate: computed('publishedAt', 'createdAt', function() {
    return this.get('publishedAt') || this.get('createdAt');
  }),

  selectedThumbnail: computed('thumbnails', 'thumbnailIndex', function() {
    return this.get('thumbnails')[this.get('thumbnailIndex')];
  }),

  thumbnailPreviewUrl: computed('preview.medium', 'thumbnails', 'thumbnailIndex', function () {
    let thumbnails = this.get('thumbnails');
    let thumbnailIndex = this.get('thumbnailIndex');

    if (typeof thumbnailIndex === 'number') {
      return thumbnails[thumbnailIndex].url;
    }

    return this.get('preview.medium');
  }),

  isUpload: computed.equal('type', UPLOAD_TYPE_PARAM),

  clearThumbnailSelection() {
    this.set('thumbnailIndex', null);
  },

  removeCustomThumbnail() {
    let previouslySelectedThumbnail = this.get('selectedThumbnail');
    let wasSelected = this.get('customThumbnail') === previouslySelectedThumbnail;

    // Set a flag for deletion on the server.
    this.set('deleteCustomThumbnail', true);

    // Remove the custom thumbnail from thumbnails array.
    this.set('thumbnails', this.get('thumbnails').filterBy('type', 'generated'));

    if (wasSelected) {
      this.selectThumbnail(this.get('thumbnails')[0]);
    } else {
      this.selectThumbnail(previouslySelectedThumbnail);
    }
  },

  selectThumbnail(thumbnail) {
    let index = this.get('thumbnails').indexOf(thumbnail);
    if (index === -1) {
      index = null;
    }

    this.set('thumbnailIndex', index);
  },

  cardPreview: computed('preview', function() {
    return this.getPreviewForSize('medium');
  }),

  // We use a computed property instead of a relationship here because
  // Ember Data expects the related data to be a part of the original request.
  similarVideos: computed(function() {
    return this.get('store').query('similar-video', { video: this.get('id') });
  }),

  viewer: computed(function() {
    return ChannelViewerModel.findOne(this.get('channel.name'));
  }),

  getPreviewForSize(size) {
    assert('Invalid preview image size selected', PREVIEW_SIZES.indexOf(size) !== -1);

    let previewOptions = this.get('preview');
    if (typeof previewOptions === 'object') {
      return previewOptions[size];
    }

    return previewOptions;
  }
});
