import Route from 'ember-route';
import env from 'web-client/config/environment';
import injectScript from 'web-client/utilities/inject-script';
import injectService from 'ember-service/inject';
import RSVP from 'rsvp';

export default Route.extend({
  api: injectService(),
  bits: injectService(),
  bitsTags: injectService(),
  gameSettings: injectService(),
  subscriptions: injectService(),
  bounties: injectService(),

  _hasSubBadges: false,
  _isBitsOnboarded: false,
  _isPayable: false,
  _areCheersPinned: false,
  _isStreamlineBountyAccepted: false,
  _isGameCommerceAmendmentAccepted: false,
  _isBitsRecipientEligible: false,

  model() {
    let api = this.get('api');
    let channelLogin = this.modelFor('dashboards').id;
    let product = this.modelFor('dashboards.subscription-dash');

    let payoutEntityRequest = api.request('get', `/api/channels/${ channelLogin }/payout_entity`);
    let payoutEntity = payoutEntityRequest.catch(jqXHR => {
      if (this.isDestroyed) { return; }

      // Staff will receive 403's for other users' payout entities; this is fine
      if (jqXHR.status === 403) { return; }

      return RSVP.reject(jqXHR);
    });

    return RSVP.hash({ payoutEntity, product });
  },

  afterModel(resolvedModel) {
    this._super(...arguments);
    let payoutPromise = this._getPayoutPayableStatus();
    let promises = [payoutPromise];

    if (resolvedModel.product && resolvedModel.product.get('isLoaded')) {
      promises.push(this._getBitsChannelSettings());
      promises.push(this._getSubBadges());
      promises.push(this._getStreamlineBountiesAmendmentsStatus());
      promises.push(this._getGameCommerceAmendmentStatus());
    }

    return RSVP.all(promises).then(() => {
      if (this.get('gameSettings.isFlWhitelisted')) {
        return injectScript(`${ env.remoteHostports.FEATURES_FILEPATH }/game-commerce.js`);
      }
    });
  },

  setupController(controller, model) {
    this._super(controller, model);

    let bitsTags = this.get('bitsTags');
    let channelID = this.modelFor('user')._id;
    let isBitsTagsEnabled = bitsTags.isChannelTagEnabled(channelID); // synchronous array look-up

    controller.set('hasSubBadges', this._hasSubBadges);
    controller.set('isBitsOnboarded', this._isBitsOnboarded);
    controller.set('isBitsTagsEnabled', isBitsTagsEnabled);
    controller.set('isPayable', this._isPayable);
    controller.set('areCheersPinned', this._areCheersPinned);
    controller.set('isStreamlineAmendmentAccepted', this._isStreamlineBountyAccepted);
    controller.set('isGameCommerceAmendmentAccepted', this._isGameCommerceAmendmentAccepted);
    controller.set('isBitsRecipientEligible', this._isBitsRecipientEligible);
  },

  _getBitsChannelSettings() {
    let bits = this.get('bits');
    let channelID = this.modelFor('user')._id;
    let bitsChannelSettingsRequest = bits.getChannelSettings(channelID);

    return bitsChannelSettingsRequest.then(response => {
      if (this.isDestroyed) { return; }

      // all partners are eligible for bits
      this._isBitsRecipientEligible = true;
      this._isBitsOnboarded = response.amendment_signed && response.onboarded;
      this._areCheersPinned = response.pin_recent_cheers;
    }, () => {
      return;
    });
  },

  _getPayoutPayableStatus() {
    let api = this.get('api');
    let isPayableRequest = api.request('get', '/api/payouts/is_payee_payable');

    return isPayableRequest.then(response => {
      if (this.isDestroyed) { return; }

      this._isPayable = !!response.payable;
    }, () => {
      return;
    });
  },

  _getSubBadges() {
    let subscriptions = this.get('subscriptions');
    let channelID = this.modelFor('user')._id;
    let subBadgesRequest = subscriptions.getBadges({ channelID });

    return subBadgesRequest.then(badges => {
      if (this.isDestroyed) { return; }

      this._hasSubBadges = badges.length > 0;
    }, () => {
      return;
    });
  },

  _getStreamlineBountiesAmendmentsStatus() {
    let bounties = this.get('bounties');
    let channelID = this.modelFor('user')._id;

    let streamlineBountyAmendment = bounties.getStreamlineBounties(channelID);

    return streamlineBountyAmendment.then(amendmentStatus => {
      if (this.isDestroyed) { return; }

      this._isStreamlineBountyAccepted = amendmentStatus.amendment_accepted;
    }, () => {
      return;
    });
  },

  _getGameCommerceAmendmentStatus() {
    let bounties = this.get('bounties');
    let channelID = this.modelFor('user')._id;

    let gameCommerceAmendment = bounties.getGameCommerceAmendmentAcceptance(channelID);

    return gameCommerceAmendment.then(amendmentStatus => {
      if (this.isDestroyed) { return; }

      this._isGameCommerceAmendmentAccepted = amendmentStatus.amendment_accepted;
    }, () => {
      return;
    });
  }
});
