import ApplicationSerializer from './application';
import DeprecatedGameModel from '../models/deprecated-game';

// convert our custom response to jsonapi:
//
//   http://jsonapi.org/format/#document-structure
//

let _makeBox = function (game) {
  return {
    type: 'sized-image',
    id: `game-${game._id}-box`,
    attributes: game.box
  };
};

let _makeLogo = function (game) {
  return {
    type: 'sized-image',
    id: `game-${game._id}-logo`,
    attributes: game.logo
  };
};

let _makeRelationships = function (box, logo) {
  return {
    box: {
      data: {
        type: 'sized-image',
        id: box.id
      }
    }, logo: {
      data: {
        type: 'sized-image',
        id: logo.id
      }
    }
  };
};

let _deserialize = function (payload) {
  let hash = {data: [], included: []};
  let data = payload;
  if (payload.top) {
    data = payload.top;
  } else if (payload.follows) {
    data = payload.follows;
  }

  data.forEach(function (opts) {
    let game = opts.game;
    game.channel_count = opts.channels;

    if (opts.viewers) {
      game.viewer_count = opts.viewers;
    }

    game.featured_on_side_panel = opts.featuredOnSidePanel;
    game.featured_on_directory = opts.featuredOnDirectory;

    if (opts.featuredOnFrontPage) {
      game.featured_on_front_page = true;
    }

    game.tracking_urls = opts.trackingUrls || null;

    let box = _makeBox(game),
        logo = _makeLogo(game),
        relationships = _makeRelationships(box, logo);

    hash.included.push(box);
    hash.included.push(logo);

    hash.data.push({
      type: 'game',
      attributes: game,
      id: game.name,
      relationships: relationships
    });

    delete game._id;
  });

  return hash;
};

export default ApplicationSerializer.extend({
  normalizeQueryResponse(store, type, payload, id) {
    let meta = this.extractMeta(store, type, payload),
        hash = _deserialize(payload);

    hash.meta = meta;
    return this._super(store, type, hash, id);
  },

  normalizeQueryRecordResponse(store, type, payload, id) {
    if (!payload) {
      return {meta: {is_response_empty: true}};
    }

    DeprecatedGameModel.create({game: payload});

    let meta = this.extractMeta(store, type, payload),
        box = _makeBox(payload),
        logo = _makeLogo(payload),
        relationships = _makeRelationships(box, logo);

    delete payload._id;

    let hash = {
      meta: meta,
      data: {
        type: 'game',
        id: payload.name,
        attributes: payload,
        relationships: relationships
      },

      included: [box, logo]
    };

    return this._super(store, type, hash, id);
  },

  /*
    Find the game that matches the target game id and then push it into the
    the game store.
  */
  normalizeFindRecordResponse(store, type, payload, id) {
    let gameRecord;

    payload.games.forEach(game => {
      if (game.name.toLowerCase() === id.toLowerCase()) {
        gameRecord = game;
      }
    });

    if (!gameRecord) {
      return { meta: { is_response_empty: true }, data: {} };
    }

    let meta = this.extractMeta(store, type, payload);
    let box = _makeBox(gameRecord);
    let logo = _makeLogo(gameRecord);
    let relationships = _makeRelationships(box, logo);

    let hash = {
      meta,
      data: {
        type: 'game',
        id: gameRecord.name,
        attributes: gameRecord,
        relationships
      },
      included: [box, logo]
    };

    return this._super(store, type, hash, id);
  }
});
