import Service from 'ember-service';
import injectService from 'ember-service/inject';
import computed from 'ember-computed';
import EmberObject from 'ember-object';
import RSVP from 'rsvp';

export default Service.extend({
  api: injectService(),

  isLoaded: false,

  // Current state for logged in channel
  isAutoHosting: false,
  snoozed: false,

  listMap: computed('list', function() {
    return this.get('list').reduce((obj, channel) => {
      obj[channel.name] = true;
      return obj;
    }, {});
  }),

  init() {
    this._super(...arguments);

    // Settings for the logged in channel
    this.set('settings', {});

    // Lists for the logged in channel
    this.set('autoHostedBy', []);
    this.set('list', []);

    RSVP.Promise.all([
      this.updateSettings(),
      this.updateStatus(),
      this.updateList(),
      this.updateAutoHostedBy()
    ]).then(() => {
      if (this.isDestroyed) { return; }

      this.set('isLoaded', true);
    });
  },

  updateSettings() {
    return this.get('api').authRequest('get', 'autohost/settings').then((payload) => {
      if (this.isDestroyed) { return; }

      this.set('settings', payload.settings);
      return payload.settings;
    });
  },

  setSettings(key, value) {
    let params = {};
    params[key] = value;
    return this.get('api').authRequest('put', 'autohost/settings', params).then((payload) => {
      if (this.isDestroyed) { return; }

      this.set('settings', payload.settings);
      return payload.settings;
    });
  },

  updateStatus() {
    return this.get('api').authRequest('get', 'autohost/status').then((payload) => {
      if (this.isDestroyed) { return; }

      this.set('isAutoHosting', payload.status.is_autohosting);
      this.set('snoozed', payload.status.snooze_duration_seconds > 0);
      return payload;
    });
  },

  snooze() {
    return this.get('api').authRequest('post', 'autohost/snooze').then((payload) => {
      if (this.isDestroyed) { return; }

      this.set('snoozed', payload.snoozed);
    });
  },

  unsnooze() {
    return this.get('api').authRequest('delete', 'autohost/snooze').then((payload) => {
      if (this.isDestroyed) { return; }

      this.set('snoozed', payload.snoozed);
    });
  },

  updateList() {
    return this.get('api').authRequest('get', 'autohost/list').then((payload) => {
      if (this.isDestroyed) { return; }

      let targets = this._emberizeChannels(payload.targets);
      this.set('list', targets);
      return targets;
    });
  },

  setList(ids) {
    if (ids.length) {
      let params = { targets: ids.join(',') };
      return this.get('api').authRequest('put', 'autohost/list', params).then((payload) => {
        if (this.isDestroyed) { return; }

        let targets = this._emberizeChannels(payload.targets);
        this.set('list', targets);
        return targets;
      });
    }

    return this.get('api').authRequest('delete', 'autohost/list').then((payload) => {
      if (this.isDestroyed) { return; }

      let targets = this._emberizeChannels(payload.targets);
      this.set('list', targets);
      return targets;
    });
  },

  addChannelToList(channelID) {
    return this.updateList().then((targetChannels) => {
      if (this.isDestroyed) { return; }

      let ids = targetChannels.map((channel) => channel._id);
      ids.push(channelID);
      return this.setList(ids);
    });
  },

  removeChannelFromList(channelID) {
    return this.updateList().then((targetChannels) => {
      if (this.isDestroyed) { return; }

      let ids = targetChannels.map((channel) => channel._id);
      let filteredIDs = ids.filter((id) => { return id !== channelID; });
      return this.setList(filteredIDs);
    });
  },

  updateAutoHostedBy() {
    return this.get('api').authRequest('get', 'autohost/list/added').then((payload) => {
      if (this.isDestroyed) { return; }

      let autoHostedBy = this._emberizeChannels(payload.added_me);
      this.set('autoHostedBy', autoHostedBy);
      return autoHostedBy;
    });
  },

  _emberizeChannels(channels) {
    return channels.map((channel) => {
      channel.displayName = channel.display_name;
      return EmberObject.create(channel);
    });
  }
});
