import Service from 'ember-service';
import computed from 'ember-computed';
import observer from 'ember-metal/observer';
import injectService from 'ember-service/inject';
import computedEqualProperty from 'web-client/utilities/computed-equal-property';

export const PUBSUB_STREAM_UP = 'stream-up';
export const PUBSUB_STREAM_DOWN = 'stream-down';

// Service for managing information about the content of the persistent player
export default Service.extend({
  session: injectService(),
  pubsub: injectService(),

  currentChannel: null,
  currentVOD: null,
  currentCollection: null,
  _pubsubStreamStatus: '',

  canGenerateClips: computed.and('hasPlayableContent', 'session.isAuthenticated'),

  willDestroy() {
    this._super(...arguments);

    let currentChannel = this.get('currentChannel');

    if (currentChannel) {
      this.get('pubsub').off(`video-playback.${currentChannel.get('name')}`);
    }
  },

  targetChannel: computed('currentChannel', 'currentChannel.hostModeTarget', 'currentVOD', function() {
    let currentVOD = this.get('currentVOD');
    if (currentVOD) {
      return currentVOD.get('channel');
    }

    return this.get('currentChannel.hostModeTarget') || this.get('currentChannel');
  }),

  hasContent: computed('currentChannel', 'currentVOD', function() {
    return this.get('currentChannel') !== null || this.get('currentVOD') !== null;
  }),

  hasPlayableContent: computed('isLive', 'isLiveHost', 'currentVOD', function() {
    let isLive = this.get('isLive');
    let isLiveHost = this.get('isLiveHost');
    let isVODContent = this.get('currentVOD') !== null;

    return isVODContent || isLive || isLiveHost;
  }),

  isHosted: computedEqualProperty('currentChannel.hostModeTarget', 'targetChannel'),

  isLive: computed('_isLiveAccordingToKraken', '_pubsubStreamStatus', 'currentChannel.playerIsLive', function() {
    if (!this.get('currentChannel')) {
      return false;
    }

    let isLivePlayer = this.get('currentChannel.playerIsLive');
    if (isLivePlayer !== undefined) {
      return isLivePlayer;
    }

    let pubsubStatus = this.get('_pubsubStreamStatus');
    if (pubsubStatus === PUBSUB_STREAM_UP) {
      return true;
    } else if (pubsubStatus === PUBSUB_STREAM_DOWN) {
      return false;
    }

    return this.get('_isLiveAccordingToKraken');
  }),

  isLiveHost: computed('currentChannel.hostModeTarget.playerIsLive', function() {
    let hostModeTarget = this.get('currentChannel.hostModeTarget');

    if (!hostModeTarget) {
      return false;
    }

    return hostModeTarget.get('playerIsLive');
  }),

  attachPubsubListener: observer('currentChannel', function() {
    let newChannel = this.get('currentChannel');
    let pubsub = this.get('pubsub');

    if (this._previousChannel !== newChannel) {
      if (newChannel) {
        pubsub.on(`video-playback.${newChannel.get('name')}`, this, this._handlePubsub);
      }

      if (this._previousChannel) {
        pubsub.off(`video-playback.${this._previousChannel.get('name')}`);
      }
    }

    this._previousChannel = newChannel;
  }),

  _isLiveAccordingToKraken: computed('currentChannel.stream', 'currentChannel.stream.isLoaded', function() {
    if (!this.get('currentChannel') || !this.get('currentChannel.stream.isLoaded')) {
      return false;
    }

    return !this.get('currentChannel.stream.isOffline');
  }),

  _handlePubsub(payload) {
    switch (payload.type) {
      case PUBSUB_STREAM_UP:
      case PUBSUB_STREAM_DOWN:
        this.set('_pubsubStreamStatus', payload.type);
        break;
    }
  }
});
