/* globals Twitch, i18n */

import { API_VERSION } from 'web-client/services/communities';
import $ from 'jquery';

const ERROR = i18n('Error');
const THANK_YOU = i18n('Thank you for your report.');
const COMMUNITY_RULES_VIOLATION = 'community_rules_violation';

function reportFormAjax() {
  $('#subwindow_close').click(function (e) {
    e.preventDefault();
    $(this).trigger('overlay.hide');
  });
  $('#report_form').on('submit', function (e) {
    e.preventDefault();
    $('#report_form_spinner').show();
    $('#report_form_submit').attr('disabled', 'disabled');

    // Check if the report reason is community_rules_violation
    let reason = $('select[name=reason]').val();
    if (reason === COMMUNITY_RULES_VIOLATION) {
      let channelName = $('input[name=reported_user]').val();
      if (channelName) {
        _getChannelId(channelName);
      }
    } else {
      $.post('/user/report', $(this).serialize())
      .done(function (response) {
        $('#report_form').trigger('overlay.hide');
        Twitch.notify.success(response);
      })
      .fail(function (response) {
        $('#report_form_spinner').hide();
        $('#report_form_submit').attr('disabled', false);
        $('#login_form_error').text(response.responseText).show();
      });
    }
  });
}

// Show error message
function _showError(message) {
  $('#report_form_spinner').hide();
  $('#report_form_submit').attr('disabled', false);
  $('#login_form_error').text(message).show();
}

// Get channel ID from channel name
function _getChannelId(channelName) {
  Twitch.api.get(`/api/channels/${channelName}`)
  .done(function(channelInfo) {
    _getCommunityId(channelInfo);
  })
  .fail(function() {
    _showError(ERROR);
  });
}

// Get community ID from channel ID
function _getCommunityId(channelInfo) {
  if (!channelInfo || !channelInfo._id) {
    _showError(ERROR);
    return;
  }
  Twitch.api.get(`/kraken/channels/${channelInfo._id}/community`, {}, { version: API_VERSION })
  .done(function(communityInfo) {
    _reportCommunityChannel(communityInfo, channelInfo);
  })
  .fail(function() {
    _showError(ERROR);
  });
}

// Send report to the communities report_channel endpoint
function _reportCommunityChannel(communityInfo, channelInfo) {
  if (!communityInfo || !communityInfo._id) {
    _showError(ERROR);
    return;
  }
  let data = {
    channel_id: channelInfo._id
  };
  Twitch.api.post(`/kraken/communities/${communityInfo._id}/report_channel`, data, { version: API_VERSION })
  .done(function() {
    $('#report_form').trigger('overlay.hide');
    _trackCommunityReport(communityInfo, channelInfo);
    Twitch.notify.success(THANK_YOU);
  })
  .fail(function() {
    _showError(ERROR);
  });
}

// Track community_moderation_report event in spade
function _trackCommunityReport(communityInfo, channelInfo) {
  let { userData } = Twitch.user.peek();
  let trackingObject = {
    community_id: communityInfo._id,
    community_name: communityInfo.name,
    url: window.location.href,
    login: userData.login,
    is_subadmin: userData.is_admin,
    is_admin: userData.is_staff,
    channel: channelInfo.name,
    reason: COMMUNITY_RULES_VIOLATION
  };
  Twitch.tracking.spade.trackEvent('community_moderation_report', trackingObject);
}

export {
  reportFormAjax
};
