import { ActiveModelSerializer } from 'ember-cli-mirage';

export default ActiveModelSerializer.extend({
  serialize(model, request) {
    let json;

    if (request.method === 'PUT') {
      json = this.serializeForUpdate(model);
    } else if (request.method === 'GET') {
      if (this.isCollection(model)) {
        json = this.serializeCollection(model);
      } else {
        json = this.serializeModel(model);
      }
    } else if (request.method === 'POST') {
      json = this.serializeForCreate(model);
    }

    return json;
  },

  // Update responses have many fewer attributes than others.
  serializeForUpdate(model) {
    let attrs = model.attrs;

    return {
      description: attrs.description,
      game: attrs.game,
      preview: serializePreviewURLs(model),
      tag_list: attrs.tag_list,
      thumbnails: serializeThumbnails(model),
      title: attrs.title
    };
  },

  // Create responses include an `upload` key
  serializeForCreate(model) {
    let video = this.serializeModel(model);

    return {
      upload: {
        url: `https://uploads.twitch.tv/upload/v${model.id}`,
        token: 'some-json-token'
      },
      video
    };
  },

  serializeModel(model) {
    let attrs = model.attrs;
    let previews = attrs.preview || serializePreviewURLs(model);
    let thumbs = attrs.thumbnails || serializeThumbnails(model);

    let json = {
      _id: `v${model.id}`,
      broadcast_id: attrs.broadcast_id,
      broadcast_type: attrs.broadcast_type,
      created_at: attrs.created_at,
      description: attrs.description,
      game: attrs.game,
      length: attrs.length,
      is_muted: attrs.is_muted,
      preview: previews,
      recorded_at: attrs.published_at,
      published_at: attrs.published_at,
      status: attrs.status,
      tag_list: attrs.tag_list,
      thumbnails: thumbs,
      title: attrs.title,
      url: `/v/${model.id}`,
      views: attrs.views
    };

    if (model.channel) {
      json['channel'] = { name: model.channel.name, display_name: model.channel.display_name };
    }

    return json;
  },

  serializeCollection(collection) {
    return {
      _total: collection.models.length,
      _links: {
        next: "http://api.twitch.tv/kraken/videos/top?limit=2&offset=2",
        self: "http://api.twitch.tv/kraken/videos/top?limit=2&offset=0"
      },
      videos: collection.models.map((video) => this.serializeModel(video))
    };
  }
});

export function thumbnailPreviewURL(model) {
  if (model.customThumbnail && model.customThumbnail.image) {
    let data = model.customThumbnail.attrs;
    return `https://static-cdn.jtvnw.net/s3_vods/${model.id}/thumb/${data.crop_x}-${data.crop_y}-${data.crop_h}-${data.crop_w}-${data.image.size}.jpeg`;
  }

  return '';
}

function serializePreviewURLs(model) {
  let thumbnailURL = thumbnailPreviewURL(model);

  if (thumbnailURL) {
    return {
      small: thumbnailURL,
      medium: thumbnailURL,
      large: thumbnailURL,
      template: 'https://www.twitch.tv/images/xarth/template_{width}x{height}.png'
    };
  }

  return {
    small: 'https://www.twitch.tv/images/xarth/404_processing_80x45.png',
    medium: 'https://www.twitch.tv/images/xarth/404_processing_320x180.png',
    large: 'https://www.twitch.tv/images/xarth/404_processing_640x360.png',
    template: 'https://www.twitch.tv/images/xarth/404_processing_{width}x{height}.png'
  };
}

function serializeThumbnails(model) {
  let thumbnailURL = thumbnailPreviewURL(model);

  if (thumbnailURL) {
    return {
      small: [{ type: 'custom', url: thumbnailURL }],
      medium: [{ type: 'custom', url: thumbnailURL }],
      large: [{ type: 'custom', url: thumbnailURL }],
      template: [{ type: 'custom', url: 'https://www.twitch.tv/images/xarth/template_{width}x{height}.png' }]
    };
  }

  return {
    small: [],
    medium: [],
    large: [],
    template: []
  };
}
