## Testing Twitch

The web-client codebase is written using [Ember](http://emberjs.com/), which
also provides an idiomatic path for testing JavaScript application code.
*It is strongly recommended that you read the [testing](https://guides.emberjs.com/v2.6.0/testing/)
section of the Ember guides* before undertaking any significant development.

#### Testing overview

There are five categories of testing we talk about for web-client. Two are
unrelated to Ember:

* Smoke testing. This is when you boot the app locally or on a staging env
  then click around ("looking for smoke").
* Smoca test. Smoca is [a set of Selenium scripts](https://git-aws.internal.justin.tv/qa/smoca)
  run on demand via [a Jenkins job](http://jenkins-master-0.prod.us-west2.justin.tv/job/qa-smoca/).

Three categories of testing happen directly in the web-client repo, and these
are the main focus of this guide.

* Unit testing asserts a unit (e.g. class, factory, object) is correct in its
  internal implementation. Unit tests are often said to assert algorithmic
  complexity. Ember provides excellent [unit testing guides](https://guides.emberjs.com/v2.6.0/testing/unit-testing-basics/),
  however their examples sometimes leak into the domain of integration
  testing. Unit tests often use the `module` or `moduleFor` function to
  setup their testing environment.
* Integration testing asserts the interfaces between units are correct. For
  example confirming a service makes an expected AJAX call is an integration
  test. So is rendering a component and seeing how the renderer processes
  it given certain arguments. Ember provides
  [integration testing guides](https://guides.emberjs.com/v2.6.0/testing/testing-components/)
  for components. We use a similar system as Ember's component testing setup
  for helpers. Integration tests may use `moduleFor` with the `needs` or
  `integration` option, `moduleForModel`, or `moduleForComponent` to setup
  their test environment.
* Acceptance testing exercises all of an application as a user would.
  Ember again provides excellent [acceptance testing guides](https://guides.emberjs.com/v2.6.0/testing/acceptance/).
  In these tests
  web-client often uses [ember-cli-mirage](http://www.ember-cli-mirage.com/) to
  stub API behavior.

As you may have noted, our definitions of unit, integration, and acceptance
testing break a little with Ember's conventional categories (which are mostly
based on the type of unit). We think the traditional definitions of these
terms are best, and in the future we hope to nudge the Ember testing
setup toward the same usages.

Web-client tests run on every PR, and can easily be run locally. Tests will execute
in [PhantomJS](http://phantomjs.org/) by default, although running them in
any browser is not very difficult. To run web-client tests:

```
// Run the tests as a one-shot parallel test run, akin to the CI server
yarn test

// Run the tests as a one-shot parallel test run in Chrome
yarn test-chrome

// Run a testing server that re-runs tests when a file changes.
yarn test-chrome:server

// Run a testing server that you can attach to a Chrome instance via a URL
ember test -server --no-launch

// Run tests with the word "api" in them
ember test --filter "api"

// Run tests in the "service:api" module
ember test --module 'service:api'

// Run a testing server on 0.0.0.0 instead of localhost, helpful when using VMs
ember test -server --no-launch -h 0.0.0.0
```

#### Using generators

Additionally, Ember will helpfully generate a test for most files automatically.
For example:

```
$ ember generate component fun-time
installing component
  create app/components/fun-time.js
  create app/templates/components/fun-time.hbs
installing component-test
  create tests/integration/components/fun-time-test.js
```

Generating `component-test` would create the test alone. You can run
`ember help generate` to see full list of available generators.

#### Understanding the directory layout

In the `tests/` directory, any file ending in `-test.js` will be evaluated
for testing. Files without that suffix are helpers for testing, such as
`signIn()` or a [Mirage](http://www.ember-cli-mirage.com/) factory.

```
tests/
  acceptance/     <-- test a whole page
  helpers/        <-- helpers for testing
  integration/
    components/   <-- component tests
    helpers/      <-- helper tests
  pages/          <-- page objects, used to abstract classes an other
                      implementation details from a test
  unit/
    mixins/       <-- mixin tests
    models/       <-- model tests (both Ember-Data and legacy)
    services/     <-- service tests
    utilities/    <-- utility tests
```

#### Using QUnit

Ember tests run in the [QUnit](https://qunitjs.com/) harness. QUnit tests
are organized into modules. For example here is a simple QUnit test of
a utility function:

```js
import { someMethod } from 'web-client/utils/example';
import { module, test } from 'qunit';

module('Unit | Utility | example');

test('#someMethod returns 3', function(assert) {
  let result = someMethod();
  assert.equal(result, 3, '3 is returned');
});
```

QUnit provides [a variety of assertions](http://api.qunitjs.com/category/assert/)
that are available as methods on the argument passed to a test.

When a QUnit test is async, there are several ways to ensure the test pauses
while async occurs.

* Acceptance tests in Ember use "async helpers" (more on them below). These
  tests automatically handle async for you.
* In other tests you should use one of the following methods:
  * Return a promise. Often in integration tests you may `import wait from 'ember-test-helpers/wait';`
    and `return wait();` at the end of your test. Because `wait()` returns
    a promise, the test will hang. `wait` operates with the same system that
    acceptance tests in Ember use, and pauses until all async Ember knows
    about (ajax, etc) has completed.
  * QUnit allows a test to be paused. For example call `let done = assert.async();`
    and when the async is complete call `done();` to make the test resume.
    [See the QUnit `async` docs](http://api.qunitjs.com/async/).

#### Debugging test failures

Sometimes a test may pass in isolation but fail when running the whole test
suite. This is often because some part of the app is not being torn down
completely in between tests. For example, there could be an event handler
still in memory for a DOM element which has already been removed from the
page, or there could be an AJAX or WebSocket request which is completing and
running code after the app has been torn down at the end of the test. A few
ways to mitigate that are to:

 1. Always tear down event handlers when you no longer need them. Use the
    `addEventListener` helper to make this automatic.
 2. Wrap all async code in a run loop and check if the context is destroyed
    before continuing. We have context-bound helpers such as `runTask` which
    help with this.
 3. Always wrap AJAX requests in an `Ember.RSVP.Promise`. This will cause
    each test to wait until all requests are complete *before* shutting down
    the app and moving on to the next test. You can use `service:api` to do
    this automatically.

If a test is failing, you need to track down the line of code which is causing
the problem. It could be completely unrelated to the code you are testing, but
the problem still needs to be fixed! Here are some tips:

 1. By default, QUnit catches all exceptions and logs them as a test failure.
    This is not helpful for finding the exact cause because it will prevent
    the stack trace from appearing in the console. To disable, **check the
    "No try-catch" checkbox** at the top of your test window, or add the
    `notrycatch` query param to the end of your test URL.
 2. Next, you can configure the Chrome Dev Tools to
    [Pause on exceptions](http://stackoverflow.com/questions/29263440/how-to-turn-on-pause-on-uncaught-exceptions-in-google-chrome-canary).
    Now your tests will pause exactly where the exception is thrown. If
    you're lucky, it will be obvious that you're in some async code that
    should have been wrapped in a Run Loop or a handler that should have been
    destroyed.
 2. It might be that the test fails when run immediately after the preceeding
    test. This is an indication that the 1st test is passing but leaving some
    data around which is corrupting the next test. You can run QUnit on
    exactly 2 or more tests to debug the issue by specifying each test ID in
    the URL, for example:
    `http://localhost.twitch.tv:7357/5387/tests/index.html?notrycatch&testId=778fa679&testId=5b5ba0e6`

### Unit testing

**Unit tests in Ember are usually written for services, mixins or utilities. You
may also see unit tests for models, helpers and components, although we generally
avoid this.**

#### Writing a mixin test

Mixins define a behavior applied to many classes. To test a mixin, it must
be applied to a class. Different mixins expect to be applied to different base
classes, so there is no one way forward. In general where
`Ember.Object` is used below it may be appropriate to use a different
class.

```js
// tests/integration/services/chat-test.js
import Ember from 'ember';
import ExampleMixin from 'web-client/mixins/example';
import { module, test } from 'qunit';

const { run, Object: EObject } = Ember;

module('context-bound-tasks', {
  beforeEach() {
    // The base class should be chosen as is appropriate. Ember.Object
    // may not be correct for all mixins.
    this.BaseObject = EObject.extend(ExampleMixin);
  },

  // This implementation of `subject` replaces the one normally at
  // `this.subject()` in tests.
  subject() {
    return this.BaseObject.create(...arguments);
  }
});

test('it is createable', function (assert) {
  let subject = this.subject();
  assert.ok(subject);
});

test('it can have a property set', function (assert) {
  let subject = this.subject();
  let value = 'a value';
  run(() => {
    subject.set('prop', value);
  });
  assert.equal(subject.get('prop'), value, 'value is set');
});
```

#### Writing a utility test

In Ember, a "utility" is any JavaScript code not related to the framework
and its resolver. A utility is tested like any unit in plain JavaScript.

For example:

```js
import { someMethod } from 'web-client/utils/example';
import { module, test } from 'qunit';

module('Unit | Utility | example');

test('#someMethod returns 3', function(assert) {
  let result = someMethod();
  assert.equal(result, 3, '3 is returned');
});
```

### Integration testing

**Integration tests in Ember are very common, and usually written for
models, serializers, adapters, services, components and helpers.**

Integration tests may use the `moduleForComponent` helper
for components and helpers. There excellent [Ember guide on component testing](https://guides.emberjs.com/v2.6.0/testing/testing-components/)
which you should review along with this document.

#### Writing a component test

Component tests are the one of the most common tests you will write in an
Ember app. The component abstraction is where most well-defined APIs in an
Ember app live, so testing at this level provides high value without the
test being brittle.

A component can be generated with the `component` blueprint:

```
$ ember generate component fun-time
installing component
  create app/components/fun-time.js
  create app/templates/components/fun-time.hbs
installing component-test
  create tests/integration/components/fun-time-test.js
```

And alternatively you can generate only the test with the `component-test`
blueprint.

A basic component test looks like this:

```js
import { moduleForComponent, test } from 'ember-qunit';
import hbs from 'htmlbars-inline-precompile';

moduleForComponent('share-button', 'ShareButtonComponent', {
  integration: true
});

test('it renders', function (assert) {
  let text = 'Share this great stream!';
  this.set('text', text);
  this.render(hbs`{{share-button message=text}}`);

  let renderedText = this.$().text();
  assert.ok(renderedText.indexOf(text) !== -1, 'template renders');
});
```

Component integration tests assert the behavior of a rendered template. Above,
the template being run is `{{share-button message=text}}`. The name of
the testing module itself doesn't control anything.

Testing a rendered template allows us to exercise the full capabilities of
components in test. For example we can call a component in block mode:

```js
this.render(hbs`
  {{#share-button}}Share this great stream!{{/share-button}}
`);
```

Or we can use helpers and other components:

```js
this.render(hbs`
  {{#share-button-group as |group|}}
    {{share-button
        on-share=group.share
        message=(capitalize 'Share this great stream!')}}
  {{/share-button-group}}
`);
```

After and of these `render` calls, use `this.$()` to get a jQuery selector
with the element wrapping the test. This can be used for assertions.

##### Updating values

Values set on the context of a test can be used in the template. For example:

```js
test('it renders', function (assert) {
  let text = 'Share this great stream!';
  this.set('text', text);
  this.render(hbs`{{share-button message=text}}`);

  let renderedText = this.$().text();
  assert.ok(renderedText.indexOf(text) !== -1, 'template renders');
});
```

They can also be updated, and the result of a rerender can be asserted:

```js
test('it renders', function (assert) {
  let text = 'Share this great stream!';
  this.set('text', text);
  this.render(hbs`{{share-button message=text}}`);

  let renderedText = this.$().text();
  assert.ok(renderedText.indexOf(text) !== -1, 'template renders');

  text = 'Or perhaps not...';
  this.set('text', text);

  renderedText = this.$().text();
  assert.ok(renderedText.indexOf(text) !== -1, 'template re-renders');
});
```

##### Testing user interaction and actions

To emit clicks and other events normally caused by a user, simply use jQuery methods.
For example:

```js
this.$().click();
this.$('.some-class button').click();
this.$('.some-class button').focus();
```

To assert an action was called, ember's bubbling action system can be used with `on`:


```js
test('it renders', function (assert) {
  assert.expect(2);
  this.on('share', arg => {
    assert.ok(true, 'share action was sent');
    assert.ok(arg.someExpectedThing, 'expected argument is present');
  });
  this.render(hbs`{{share-button on-share='share'}}`);
  this.$().click();
});
```

Many modern components eschew the bubbling action setup in favor of improved
actions, sometimes called closure actions. [More in the Ember API docs on
the action helper](http://emberjs.com/api/classes/Ember.Templates.helpers.html#method_action).
For these kinds of actions passing a function directly to a component or
helper is sufficient, and using `on` is unnecessary.

For example without a component:

```js
test('it renders', function (assert) {
  assert.expect(2);
  this.set('share', arg => {
    assert.ok(true, 'share action was sent');
    assert.ok(arg.someExpectedThing, 'expected argument is present');
  });
  this.render(hbs`<button onclick={{action share}}>`);
  this.$().click();
});
```

And similarly with a component:

```js
test('it renders', function (assert) {
  assert.expect(2);
  this.set('share', arg => {
    assert.ok(true, 'share action was sent');
    assert.ok(arg.someExpectedThing, 'expected argument is present');
  });
  this.render(hbs`{{share-button on-share=(action share)}}`);
  this.$().click();
});
```

##### Managing dependencies

When performing an integration test, it may be useful to have access to a
dependency normally injected by Ember. For example:

```js
import { moduleForComponent, test } from 'ember-qunit';
import hbs from 'htmlbars-inline-precompile';
import Ember from 'ember';

const {run} = Ember;

moduleForComponent('when-wide', {
  beforeEach() {
    this.inject.service('dimensions', {as: 'dimensionsService'});
  },
  integration: true
});

test('renders when wide', function (assert) {

  this.render(hbs`{{#when-wide}}Only rendered when wide{{/when-wide}}`);
  assert.equal(this.$().text().trim(), '', 'precond - nothing rendered by default');

  let service = this.get('dimensionsService');
  run(() => {
    service.set('isWide', true);
  });

  assert.equal(this.$().text().trim(), 'Only rendered when wide', 'renders when wide');
});
```

In other cases, you may want to completely replace the service with a mocked
version. For example:

```js
import { moduleForComponent, test } from 'ember-qunit';
import hbs from 'htmlbars-inline-precompile';
import Ember from 'ember';

const {run, Service} = Ember;

const MockService = Service.extend({
  isWide: true
});

moduleForComponent('when-wide', {
  beforeEach() {
    this.register('service:dimensions', MockService);
  },
  integration: true
});

test('renders when wide', function (assert) {
  this.render(hbs`{{#when-wide}}Only rendered when wide{{/when-wide}}`);
  assert.equal(this.$().text().trim(), 'Only rendered when wide', 'renders when wide');
});
```

#### Writing a helper test

Helpers should also be tested using the `moduleForComponent` helper, and should
be in the `tests/integration/helpers/` directory. *At this time, the
helper test generator does not create helper tests as integration tests*.
Thus you will need to create the test file yourself.

Because the APIs use for testing helpers are basically the same as those
for testing components, you should read this document first.

An example helper integration test looks like this:

```js
tests/integration/helpers/is-blue-test.js
import { moduleForComponent, test } from 'ember-qunit';
import hbs from 'htmlbars-inline-precompile';

moduleForComponent('is-blue', 'helper:is-blue', {
  integration: true
});

test('is true when blue', function (assert) {
  this.set('color', 'magenta');
  this.render(hbs`{{if (is-blue color) 'is-blue' 'not-blue'}}`);
  assert.equal(this.$().text(), 'not-blue');

  this.set('color', 'blue');
  assert.equal(this.$().text(), 'is-blue');
});
```

The techniques for updating values and managing dependencies from the
section on components apply to helpers as well.

#### Writing an Ember-Data model test

web-client, at this writing, sports two model types. The first are [Ember-Data](https://github.com/emberjs/data)
models. In time all models in the web-client codebase will be ported to this
system.

Model unit tests should use the `moduleForModel` method. For example:

```js
// tests/unit/models/foo-test.js
import { moduleForModel, test } from 'ember-qunit';

moduleForModel('foo', 'Unit | Model | foo');

test('it exists', function(assert) {
  let model = this.subject();
  assert.ok(!!model);
});
```

`subject()` takes the argument of initial values passed to that model. This
form of test works well for asserting the behavior of a CP or method
on the model class.

Often a model test will require using the Ember store to exercise other
parts of the data layer in integration. The `store` method will allow you to
call a number of methods that pertain to more than a single model. For
example, this code can assert the behavior of a local collection of records:

```js
// tests/unit/models/foo-test.js
import Ember from 'ember';
import { moduleForModel, test } from 'ember-qunit';

const { run } = Ember;

moduleForModel('foo', 'Unit | Model | foo');

test('it can be peeked', function(assert) {
  let store = this.store();
  run(() => {
    store.push('foo', {
      title: 'some foo title'
    });
  });
  let records = store.peekAll('foo');
  assert.equal(records.get('length'), 1, 'one record present');
});
```

If your model is dependent on other models through relationships or if you
want to test adapter and serializer behavior via integration, you may be
required to pass a list of factories to `needs`:

```js
moduleForModel('foo', 'Unit | Model | foo', {
  needs: ['model:other-model', 'adapter:foo']
});
```

Or alternatively you can remove isolation from the module with `integration: true`:

```js
moduleForModel('foo', 'Unit | Model | foo', {
  integration: true
});
```

#### Mocking APIs with Mirage

Integration tests are often used to assert how a model or other Ember
factory interacts with the network. For example, many of the Ember-Data
models in web-client rely on customized adapter and serializer behaviors-
testing those as part of an integration is a non-brittle way to assert
that behavior (compared to unit tests of the serializer, for example).

In web-client, [ember-cli-mirage](http://www.ember-cli-mirage.com/) is used
to stub and mock AJAX requests. Mirage provides a "server" object that
acts like a database of records on the API webserver. The actual translation of
records in that database (fixtures) to API endpoints, happens in `mirage/config.js`
and `mirage/routes/`.

Often you can rely on pre-existing routes and simply use factories
appropriate to your test. For example:

```js
// tests/integration/models/foo-test.js
import { moduleForModel, test } from 'ember-qunit';
import {
  setup as setupMirage,
  teardown as teardownMirage
} from '../../helpers/setup-mirage-for-integration';

moduleForModel('foo', 'Unit | Model | foo', {
  integreation: true,
  beforeEach() {
    setupMirage(this);
  },
  afterEach() {
    teardownMirage(this);
  }
});

test('it can be fetched', function(assert) {
  let done = assert.async();
  let name = 'somename';
  let id = 'some-id';
  this.server.create('foo', {id, name});
  let store = this.store();
  store.fetchRecord('foo', id).then(record => {
    assert.equal(record.get('name'), name, 'name as returned from server');
  }).finally(done);
});
```

Given an appropriate factory `mirate/factories/foo.js` and a route mapping
the URL correctly to that factory, this test would be valid.

When learning about mirage, be sure you read the 0.2.0 documentation.

#### Writing a legacy model test

web-client contains many models and collections not based on Ember-Data. These
are based on the `app/models/store.js` defined base class. Though we've
ported some feature to them like support for [Ember dependency injection](https://guides.emberjs.com/v2.6.0/applications/dependency-injection/)
they remain a custom part of the web-client codebase.

To test these classes, simply use `module` and test them like a unit. `module`
is the most basic way to define a QUnit testing module, and it bring no
special behavior with it.

```js
// tests/integration/models/foo-test.js
import { module, test } from 'qunit';
import FooModel from 'web-client/models/foo';
import {
  setup as setupMirage,
  teardown as teardownMirage
} from '../../helpers/setup-mirage-for-integration';

module('Integration | Model | foo', {
  beforeEach() {
    setupMirage(this);
  },
  afterEach() {
    teardownMirage(this);
  }
});

test('loading a foo', function(assert) {
  let done = assert.async();
  let id = 'some-id';
  let title = 'some title';
  let foo = FooModel.create({ id });

  this.server.create('foo', {id, title});

  foo.load().then(() => {
    assert.equal(foo.get('title'), title, 'title is set');
  }).finally(done);
});
```

The guidance around API mocking and async provided for Ember-Data models should
apply to these bespoke models as well.

#### Writing a service test

Service tests should use `moduleFor`. This module type has a `this.subject()`
which is an instance of which ever module the test is referencing. It is
similar to `moduleForModel`, but more generic. For example:

```js
// tests/integration/services/chat-test.js
import { moduleFor, test } from 'ember-qunit';

moduleFor('service:chat', 'Integration | Service | chat', {
  integration: true
});

test('it exists', function(assert) {
  let service = this.subject();
  assert.ok(service);
});
```

As with `moduleForModel`, you may pass `needs` or `integration` as an option
to allow this test to access other factories. Service tests are not always
integration tests, however they often coordinate multiple other factories
and thus integration tests are appropriate.

### Acceptance testing

Ember acceptance testing exercises a running application in browser. Between
tests, the application is destroyed and fresh one created for the next
test.

You should read the [Ember acceptance testing guides](https://guides.emberjs.com/v2.6.0/testing/acceptance/)
to get a good overview of how acceptance testing works.

Acceptance tests will use the `moduleForAcceptance` method for setup, and will
be stored at `tests/acceptance/` in the project.

#### Using Ember test helpers

Ember provides two kinds of testing helpers. The first, asynchronous test
helpers, allow you to write async-safe code in a synchronous manner. For example:

```js
import { test } from 'qunit';
import moduleForAcceptance from 'web-client/tests/helpers/module-for-acceptance';

moduleForAcceptance('Acceptance | login');

test('visiting /login', function(assert) {
  visit('/login');
  andThen(function() {
    assert.equal(currentURL(), '/login');
  });
  click('button.forgot-password');
  andThen(function() {
    assert.equal(currentURL(), '/forgot-password');
  });
});
```

When this test runs, it will enqueue four steps the test needs to execute. `visit, andThen callback, click, andThen callback`.
Between each step, the test runner will wait for any pending ajax or scheduled
timeouts to complete. Thus before the final `andThen` callback runs, we can
be sure the entire routing transition away from `/login` and to `/forgot-password` is complete.

Ember also provides synchronous helpers which do *not* enqueue as a step. For
example, `currentURL()` in the above code simply runs immediately like a
normal JavaScript function.

Often, the first level of indentation in an acceptance test will be async
helpers. There is no need to nest async helpers (`click` inside of `andThen`),
but sometimes a steps may be dependent on something from a previous step and
one level of nesting is permitted.

For a thorough explanation of acceptance testing, you should read the [Ember acceptance testing guides](https://guides.emberjs.com/v2.6.0/testing/acceptance/).

#### Writing a page object

Page objects are a pattern for writing DOM-level acceptance tests. This
[post by Martin Fowler](http://martinfowler.com/bliki/PageObject.html) is a good
high-level reference. In web-client we have a bespoke implementation of this
pattern.

You may want to use a page object in an integration test or in an acceptance
test to decouple your test from the implementation details of the DOM.

Page objects should inherit from the base class in `tests/pages/base.js`. For
example, this page object provides a selector for a form submission button:

```js
// tests/pages/channel.js
import Base from './base';

export default Base.extend({

  _url: computed('name', function() {
    return `/${this.get('name')}`;
  }),

  searchField() {
    return '.js-search-input';
  }

  submitButton() {
    return '.js-submit-form';
  }

});
```

And this test consumes the page object:

```js
// tests/acceptance/channel-test.js
import { test } from 'qunit';
import moduleForAcceptance from '../helpers/module-for-acceptance';
import ChannelPage from '../pages/channel';

moduleForAcceptance('Acceptance-Channel', {
  beforeEach() {
    this.channelName = 'CSGO';
    this.page = ChannelPage.create({
      name: this.channelName
    });
  }
});

test('submitting search', function (assert) {
  visit(this.page.url());
  fillIn(this.page.searchField(), 'greatplayer');
  click(this.page.submitButton());
  andThen(function () {
    assert.equal(currentURL(), '/search/greatplayer');
  });
});
```

Please refer to the base class in `tests/pages/base.js` to discover the default
features that come with a web-client page object. To see what a specific
object has as behavior, you should read its implementation (most are very
simple).
